//
//  AVLibrary.swift
//  AVCaptureDemo
//
//  Created by 朱克剛 on 2022/9/10.
//

import SwiftUI
import Foundation
import AVFoundation

class CameraManager: NSObject, AVCapturePhotoCaptureDelegate {
    static let current = CameraManager()
    let session = AVCaptureSession()
    // 前鏡頭
    private var frontCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .front
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        return nil
    }
    // 後鏡頭
    private var backCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .back
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        return nil
    }
    
    private let photoOutput = AVCapturePhotoOutput()
    override init() {
        super.init()
        if let frontCamera {
            session.addInput(frontCamera)
            session.addOutput(photoOutput)
        }
    }
    
    func getPreviewLayer() -> CALayer {
        let layer = AVCaptureVideoPreviewLayer(session: session)
        layer.name = "preview"
        layer.videoGravity = .resizeAspectFill
        return layer
    }
    
    func toggleCamera(isFront: Bool) {
        guard let frontCamera, let backCamera else {
            return
        }
        session.beginConfiguration()
        session.removeInput(session.inputs.last!)
        session.addInput(isFront ? frontCamera : backCamera)
        session.commitConfiguration()
    }
    
    // MARK: 拍照
    func takePicture() {
        if !session.isRunning {
            session.startRunning()
            // 等相機參數最佳化
            Thread.sleep(forTimeInterval: 0.3)
        }
        let setting = AVCapturePhotoSettings()
        setting.isAutoRedEyeReductionEnabled = true
        setting.flashMode = .auto
        photoOutput.capturePhoto(with: setting, delegate: self)
    }
    
    // MARK: 取得拍照完的資料
    func photoOutput(_ output: AVCapturePhotoOutput, didFinishProcessingPhoto photo: AVCapturePhoto, error: Error?) {
        session.stopRunning()
        if  let data = photo.fileDataRepresentation() {
            let uiImage = UIImage(data: data)
            UIImageWriteToSavedPhotosAlbum(uiImage!, nil, nil, nil)
        }
    }
}

class MovieManager: NSObject, AVCaptureFileOutputRecordingDelegate {
    static let current = MovieManager()
    let session = AVCaptureSession()
    private var backCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .back
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        
        return nil
    }
    private var microphone: AVCaptureDeviceInput? {
        let microphone = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInMicrophone],
            mediaType: .audio,
            position: .unspecified
        ).devices.first
        
        if let microphone {
            return try? AVCaptureDeviceInput(device: microphone)
        }
        
        return nil
    }
    
    private let movieOutput = AVCaptureMovieFileOutput()
    override init() {
        super.init()
        if let backCamera, let microphone {
            session.addInput(backCamera)
            session.addInput(microphone)
            session.addOutput(movieOutput)
        }
    }
    
    func getPreviewLayer() -> CALayer {
        let layer = AVCaptureVideoPreviewLayer(session: session)
        layer.name = "preview"
        layer.videoGravity = .resizeAspectFill
        return layer
    }
    
    // MARK: 開始錄影
    func startRecording() {
        let filename = NSTemporaryDirectory() + "tmp.mov"
        let url = URL(fileURLWithPath: filename)
        movieOutput.startRecording(to: url, recordingDelegate: self)
    }
    
    // MARK: 結束錄影
    func stopRecording() {
        movieOutput.stopRecording()
    }
    
    // MARK: 將儲存的影片檔複製到相簿
    func fileOutput(_ output: AVCaptureFileOutput, didFinishRecordingTo outputFileURL: URL, from connections: [AVCaptureConnection], error: Error?) {
        if UIVideoAtPathIsCompatibleWithSavedPhotosAlbum(
            outputFileURL.path
        ) {
            UISaveVideoAtPathToSavedPhotosAlbum(
                outputFileURL.path,
                self,
                #selector(completion),
                nil
            )
        }
    }
    
    // MARK: 刪除儲存的影片檔
    @objc func completion(_ videoPath: String, error: Error?, contextInfo: Any?) {
            try? FileManager.default.removeItem(atPath: videoPath)
    }
}

/*
class ClassicAudioManager: NSObject, AVCaptureAudioDataOutputSampleBufferDelegate {
    static let current = ClassicAudioManager()
    let session = AVCaptureSession()
    private var microphone: AVCaptureDeviceInput? {
        let microphone = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInMicrophone],
            mediaType: .audio,
            position: .unspecified
        ).devices.first
        
        if let microphone {
            return try? AVCaptureDeviceInput(device: microphone)
        }
        
        return nil
    }
    private let audioOutput = AVCaptureAudioDataOutput()
    override init() {
        super.init()
        if let microphone {
            session.addInput(microphone)
            session.addOutput(audioOutput)
            audioOutput.setSampleBufferDelegate(self, queue: .global())
        }
    }
    
    func captureOutput(_ output: AVCaptureOutput, didOutput sampleBuffer: CMSampleBuffer, from connection: AVCaptureConnection) {
        print("hello")
    }
}
*/

class AudioManager {
    static let current = AudioManager()
    private var recorder: AVAudioRecorder?
    private var player: AVAudioPlayer?
    
    private func setAudioSession() throws {
        let session = AVAudioSession.sharedInstance()
        // 支援背景錄放音以及不允許混音
        try session.setCategory(.playAndRecord)
        // 通知其他目前播放音樂程式暫停播放
        try session.setActive(true, options: .notifyOthersOnDeactivation)
    }
    
    // MARK: 開始錄音
    func startRecording(_ filename: String) {
        let path = NSHomeDirectory() + "/Documents/" + filename + ".ima4"
        let url = URL(fileURLWithPath: path)
        
        do {
            try setAudioSession()
            recorder = try AVAudioRecorder(
                url: url,
                settings: [
                    AVFormatIDKey: NSNumber(value: kAudioFormatAppleIMA4)
                ]
            )
            
            if let recorder, recorder.prepareToRecord() {
                recorder.record()
            } else {
                print("recorder is not ready")
            }
        } catch {
            print(error)
        }
    }
    
    // MARK: 停止錄音
    func stopRecording() {
        recorder?.stop()
    }
    
    // MARK: 放音
    func play(_ filename: String) {
        let path = NSHomeDirectory() + "/Documents/" + filename + ".ima4"
        let url = URL(fileURLWithPath: path)
        do {
            player = try AVAudioPlayer(contentsOf: url)
            if let player, player.prepareToPlay() {
                player.play()
            } else {
                print("player is not ready")
            }
        } catch {
            print(error)
        }
    }
}

class BarcodeResult: ObservableObject {
    @Published var data: (type: String, value: String) = ("", "")
}

class MetadataManager: NSObject, AVCaptureMetadataOutputObjectsDelegate {
    let barcoardResult = BarcodeResult()
    static let current = MetadataManager()
    let session = AVCaptureSession()
    private var backCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .back
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        
        return nil
    }
    private let metaOutput = AVCaptureMetadataOutput()
    override init() {
        super.init()
        if let backCamera {
            session.addInput(backCamera)
            session.addOutput(metaOutput)
            
            // 接受所有可識別格式
            metaOutput.metadataObjectTypes = metaOutput.availableMetadataObjectTypes
            metaOutput.setMetadataObjectsDelegate(self, queue: .global())
        }
    }
    
    func getPreviewLayer() -> CALayer {
        let layer = AVCaptureVideoPreviewLayer(session: session)
        layer.name = "preview"
        layer.videoGravity = .resizeAspectFill
        return layer
    }
    
    // 掃描到條碼後會觸發的函數
    func metadataOutput(_ output: AVCaptureMetadataOutput, didOutput metadataObjects: [AVMetadataObject], from connection: AVCaptureConnection) {
        // 停止預覽（也就是停止辨識）
        session.stopRunning()
        metadataObjects.forEach { data in
            if let data = data as? AVMetadataMachineReadableCodeObject {
                let type = data.type.rawValue
                let value = data.stringValue ?? "NULL"
                
                print("條碼型態：\(type)")
                print("條碼內容：\(value)")
                
                DispatchQueue.main.async {
                    self.barcoardResult.data = (type, value)
                }
            }
        }
    }
}

struct PreviewView: UIViewRepresentable {
    var previewLayer: CALayer
    var proxy: GeometryProxy
    
    func makeUIView(context: Context) -> UIView {
        let view = UIView()
        view.layer.addSublayer(previewLayer)
        return view
    }
    
    func updateUIView(_ uiView: UIView, context: Context) {
        let previewLayer = uiView.layer.sublayers?.first { layer in
            layer.name == "preview"
        }
        if let previewLayer {
            previewLayer.frame = proxy.frame(in: .local)
        }
    }
    
    typealias UIViewType = UIView
}
