//
//  SocketClient.swift
//  InternetDemo
//
//  Created by 朱克剛 on 2022/9/3.
//

import Foundation
import Combine
import Network

class SocketClient: ObservableObject {
    @Published var receivedString: String = ""
    @Published var isConnect = false
    private var conn: NWConnection!
    
    func start(host: String, port: Int) {
        conn = NWConnection(
            host: NWEndpoint.Host(host),
            port: NWEndpoint.Port(String(port))!,
            using: .tcp
        )
        conn.stateUpdateHandler = stateChanged(_:)
        conn.start(queue: .global())
    }

    func stateChanged(_ state: NWConnection.State) {
        switch state {
        case .ready:
            print("connection ready")
            DispatchQueue.main.async {
                self.isConnect = true
            }
            receive()
            
        case .failed(let error):
            DispatchQueue.main.async {
                self.isConnect = false
            }
            print("connection faile: \(error)")
            
        default:
            DispatchQueue.main.async {
                self.isConnect = false
            }
            print("connection state: \(state)")
        }
    }
    
    func send(_ data: Data) {
        // 送出資料
        conn.send(content: data, completion: .contentProcessed({ error in
            if let error {
                print(error.debugDescription)
                self.conn.cancel()
            }
        }))
    }

    func receive() {
        // 收到資料
        conn.receive(minimumIncompleteLength: 1, maximumLength: Int.max, completion: { completeContent, contentContext, isComplete, error in
            guard let data = completeContent else {
                if let error {
                    print(error.debugDescription)
                }
                self.conn.cancel()
                return
            }
            
            if let string = String(data: data, encoding: .utf8) {
                print("received: \(string)")
                DispatchQueue.main.async {
                    // 將收到的字串發佈出去
                    self.receivedString = string
                }
            }
            
            // 繼續等下一筆資料
            self.receive()
        })
    }
}
