//
//  SocketServer.swift
//  InternetDemo
//
//  Created by 朱克剛 on 2022/9/3.
//

import Foundation
import Network

class ConnectionManager: Identifiable {
    let id = UUID()
    private var server: SocketServer!
    private var conn: NWConnection!

    init(_ server: SocketServer,  _ connection: NWConnection) {
        self.server = server
        conn = connection
        conn.stateUpdateHandler = stateChanged(_:)
        conn.start(queue: .global())
        // 顯示client端連線資訊
        switch connection.endpoint {
        case .hostPort(let host, let port):
            print("\(host):\(port) Connect")
        default:
            break
        }
    }
    
    func disconnect() {
        server.connectionManagers.removeAll { manager in
            manager.id == id
        }
        conn = nil
    }
    
    private func stateChanged(_ state: NWConnection.State) {
        switch state {
        case .ready:
            print("connection ready")
            receive()
            
        case .failed(let error):
            print("connection faile: \(error)")
            disconnect()
            
        case .cancelled:
            print("connection cancel")
            disconnect()
            
        default:
            print("connection state: \(state)")
        }
    }
    
    func send(_ data: Data) {
        // 送出資料
        conn.send(content: data, completion: .contentProcessed({ error in
            if let error {
                print(error.debugDescription)
                self.conn.cancel()
            }
        }))
    }

    func receive() {
        // 收到資料
        conn.receive(minimumIncompleteLength: 1, maximumLength: Int.max, completion: { completeContent, contentContext, isComplete, error in
            guard let data = completeContent else {
                if let error {
                    print(error.debugDescription)
                }
                self.conn.cancel()
                return
            }
            
            let string = String(data: data, encoding: .utf8)
            print("received: \(string!)")
            
            // 範例：收到資料後原封不動送回去
            self.send(data)
            // 繼續等下一筆資料
            self.receive()
        })
    }
}

class SocketServer {
    var connectionManagers: [ConnectionManager] = []
    private var listener: NWListener!
    
    init(port: NWEndpoint.Port) {
        do {
            listener = try NWListener(using: .tcp, on: port)
            listener.stateUpdateHandler = stateChanged(_:)
            listener.newConnectionHandler = accept(connection:)
        } catch {
            fatalError(error.localizedDescription)
        }
    }
    
    func start() {
        listener.start(queue: .global())
    }
    
    private func stateChanged(_ state: NWListener.State) {
        switch state {
        case .ready:
            print("listener ready")
            
        case .failed(let error):
            fatalError("listener faile: \(error)")

        default:
            print("listener state: \(state)")
        }
    }

    
    private func accept(connection: NWConnection) {
        // 有client要求連線時, 這裡要產生一個新的ConnectionManager實體
        connectionManagers.append(ConnectionManager(self, connection))
    }
}
