//
//  CentralManager.swift
//  BLECentral
//
//  Created by 朱克剛 on 2022/9/12.
//

import Foundation
import CoreBluetooth

// 根據周邊裝置規格書所示，定義哪些characteristic想要使用
enum CharacteristicType: String {
    case C001 = "C001"
    case C002 = "C002"
}

class CentralManager: NSObject, CBCentralManagerDelegate, CBPeripheralDelegate, ObservableObject {
    @Published var c001_reply_string: String = ""
    
    static let shared = CentralManager()
    
    // 管理central端，需宣告全域
    private var centralManager: CBCentralManager?
    // 連線完成的周邊裝置
    private var connectPeripheral: CBPeripheral?
    // 儲存周邊裝置藍牙名稱
    private var peripheralName: String!
    // 儲存所有周邊裝置提供的characteristic
    private var chars = [String: CBCharacteristic]()
    
    override init() {
        super.init()
        // 搜尋到的名稱未必是peripher設定好的名字
        peripheralName = "my_ble_device"
        // 將觸發1號method
        centralManager = CBCentralManager(
            delegate: self,
            queue: .global()
        )
    }
    
    // MARK: - 1號method
    func centralManagerDidUpdateState(_ central: CBCentralManager) {
        guard central.state == .poweredOn else {
            // 若藍牙沒開啟，作業系統會自動跳出提示
            return
        }
        
        if reconnectIfNeeded() {
            // 將觸發3號method
            print("準備重新連線")
            central.connect(connectPeripheral!, options: nil)
        } else {
            // 將觸發2號method
            print("準備掃描藍牙裝置")
            central.scanForPeripherals(withServices: nil, options: nil)
        }
    }
    
    // MARK: 2號method
    func centralManager(_ central: CBCentralManager, didDiscover peripheral: CBPeripheral, advertisementData: [String : Any], rssi RSSI: NSNumber) {
        let name = peripheral.name ?? "NONAME"
        print("掃描到 \(name)")
        
        if name == peripheralName || name == "iPad" {
            // 停止掃描
            central.stopScan()
            connectPeripheral = peripheral
            peripheral.delegate = self
            
            // 將觸發3號method
            centralManager?.connect(peripheral, options: nil)
        }
    }
    
    // MARK: 3號method（連線失敗）
    func centralManager(_ central: CBCentralManager, didFailToConnect peripheral: CBPeripheral, error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
    }

    // MARK: 3號method（連線成功）
    func centralManager(_ central: CBCentralManager, didConnect peripheral: CBPeripheral) {
        chars.removeAll()
        
        // 將UUID存起來，斷線後重新連線時須使用
        UserDefaults.standard.set(
            peripheral.identifier.uuidString,
            forKey: peripheralName
        )
        
        // 將觸發4號method
        peripheral.discoverServices(nil)
    }
    
    // MARK: 4號method
    func peripheral(_ peripheral: CBPeripheral, didDiscoverServices error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
        peripheral.services?.forEach { service in
            // 將觸發5號method
            peripheral.discoverCharacteristics(nil, for: service)
        }
    }
    
    // MARK: 5號method
    func peripheral(_ peripheral: CBPeripheral, didDiscoverCharacteristicsFor service: CBService, error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
        service.characteristics?.forEach { char in
            print("找到: \(char.uuid.uuidString)")
            chars[char.uuid.uuidString] = char
        }
    }
    
    // MARK: - 訂閱要求
    func subscribe(for type: CharacteristicType) {
        guard let char = chars[type.rawValue] else {
            print("@\(#function)")
            print("Error: \(type.rawValue) not found")
            return
        }
        connectPeripheral?.setNotifyValue(true, for: char)
    }

    // MARK: 取消訂閱要求
    func unsubscribe(for type: CharacteristicType) {
        guard let char = chars[type.rawValue] else {
            print("@\(#function)")
            print("Error: \(type.rawValue) not found")
            return
        }
        connectPeripheral?.setNotifyValue(false, for: char)
    }
    
    // MARK: - 讀取要求
    func read(for type: CharacteristicType) {
        guard let char = chars[type.rawValue] else {
            print("@\(#function)")
            print("Error: \(type.rawValue) not found")
            return
        }
        connectPeripheral?.readValue(for: char)
    }
    
    // MARK: 寫入要求
    func write(for type: CharacteristicType, data: Data) {
        guard let char = chars[type.rawValue] else {
            print("@\(#function)")
            print("Error: \(type.rawValue) not found")
            return
        }
       
        // 需不需要peripheral回傳確認訊息，必須依照peripheral設計
        var writeType: CBCharacteristicWriteType? = nil
        switch type {
        case .C001:
            writeType = .withoutResponse
        case .C002:
            writeType = .withResponse
        }
        
        // 發出寫入要求
        connectPeripheral?.writeValue(
            data,
            for: char,
            type: writeType!
        )
    }
    
    // MARK: - 收到peripher傳過來的資料
    func peripheral(_ peripheral: CBPeripheral, didUpdateValueFor characteristic: CBCharacteristic, error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
        
        switch characteristic.uuid.uuidString {
        case CharacteristicType.C001.rawValue:
            if let data = characteristic.value {
                if let string = String(data: data, encoding: .utf8) {
                    // 若回傳的內容為字串，這裡將字串印出
                    print(string)
                    DispatchQueue.main.async {
                        self.c001_reply_string = string
                    }
                }
            }
            
        default:
            break
        }
    }
    
    // MARK: - 寫入後收到回覆確認
    func peripheral(_ peripheral: CBPeripheral, didWriteValueFor characteristic: CBCharacteristic, error: Error?) {
        if error != nil {
            print("@\(#function)")
            print("Peripheral 回傳「寫入要求錯誤」: \(error!)")
        } else {
            print("Peripheral 回傳「寫入要求success」")
        }
    }
}

extension CentralManager {
    // MARK: 判斷是否需要重新連線
    func reconnectIfNeeded() -> Bool {
        guard let uuidString = UserDefaults.standard.string(forKey: peripheralName) else {
            return false
        }
        
        let uuid = UUID(uuidString: uuidString)!
        let list = centralManager?.retrievePeripherals(withIdentifiers: [uuid])

        connectPeripheral = list!.first
        connectPeripheral?.delegate = self
        return true
    }
    
    // MARK: 斷線
    func centralManager(_ central: CBCentralManager, didDisconnectPeripheral peripheral: CBPeripheral, error: Error?) {
        if error != nil {
            print("@\(#function)")
            print(error!)
        }
        
        if reconnectIfNeeded() {
            print("收不到藍牙訊號斷線")
            // 將觸發 3號method
            central.connect(peripheral, options: nil)
        } else {
            // 這裡要通知使用者藍牙已經解配對了
            print("Center發出解配對指令")
        }
    }
    
    // MARK: 解配對
    func unpair() {
        // 務必提醒使用者必須從系統設定中「忘記裝置」，否則無法再配對
        UserDefaults.standard.removeObject(forKey: peripheralName)
        if let connectPeripheral {
            // 呼叫後藍牙連線中斷
            centralManager?.cancelPeripheralConnection(connectPeripheral)
        }
    }
}

