//
//  PeripheralManager.swift
//  BLEPeripheralNew
//
//  Created by 朱克剛 on 2022/9/13.
//

import Foundation
import CoreBluetooth

enum ServiceType: String {
    case SERV_A001 = "A001"
}

enum CharacteristicType: String {
    case C001 = "C001"
    case C002 = "C002"
}

class PeripheralManager: NSObject, CBPeripheralManagerDelegate {
    static let shared = PeripheralManager()
    let BLUETOOTH_NAME = "my_ble_device"
    
    // 管理peripheral，需宣告全域
    private var peripheralManager: CBPeripheralManager?

    override init() {
        super.init()
        // 將觸發1號method
        peripheralManager = CBPeripheralManager(
            delegate: self,
            queue: .main
        )
    }
    
    // MARK: - 1號method
    func peripheralManagerDidUpdateState(_ peripheral: CBPeripheralManager) {
        guard peripheral.state == .poweredOn else {
            // 若藍牙沒開啟，作業系統會自動跳出提示
            return
        }
        
        ///////////// 開始設定 service 與 characteristic ////////////
        let service = CBMutableService(
            type: CBUUID(string: ServiceType.SERV_A001.rawValue),
            primary: true
        )
        var chars = [CharacteristicType: CBMutableCharacteristic]()
        
        chars[.C001] = CBMutableCharacteristic(
            type: CBUUID(string: CharacteristicType.C001.rawValue),
            properties: [.notify, .read, .writeWithoutResponse],
            value: nil,
            permissions: [.readable, .writeable]
        )

        chars[.C002] = CBMutableCharacteristic(
            type: CBUUID(string: CharacteristicType.C002.rawValue),
            properties: [.write],
            value: nil,
            permissions: [.writeEncryptionRequired]
        )

        // 在 service 中填入 characteristic 陣列
        service.characteristics = Array(chars.values)
        // 準備觸發2號method
        peripheralManager?.add(service)
    }

    // MARK: 2號method
    func peripheralManager(_ peripheral: CBPeripheralManager, didAdd service: CBService, error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
        
        // 開始廣播，準備觸發3號method
        peripheral.startAdvertising([
            CBAdvertisementDataServiceUUIDsKey: [service.uuid],
            CBAdvertisementDataLocalNameKey: BLUETOOTH_NAME
        ])
    }
    
    // MARK: 3號method
    func peripheralManagerDidStartAdvertising(_ peripheral: CBPeripheralManager, error: Error?) {
        guard error == nil else {
            print("@\(#function)")
            print(error!)
            return
        }
        print("開始廣播")
    }

    
    // MARK: - 開始訂閱
    func peripheralManager(_ peripheral: CBPeripheralManager, central: CBCentral, didSubscribeTo characteristic: CBCharacteristic) {
        print("開始訂閱")
        DispatchQueue.global().async {
            for _ in 1...3 {
                peripheral.updateValue(
                    "Hello Central".data(using: .utf8)!,
                    for: characteristic as! CBMutableCharacteristic,
                    onSubscribedCentrals: nil)
                sleep(1)
            }
        }
    }

    // MARK: 取消訂閱
    func peripheralManager(_ peripheral: CBPeripheralManager, central: CBCentral, didUnsubscribeFrom characteristic: CBCharacteristic) {
        print("取消訂閱")
    }
    
    // MARK: - 收到central端讀取要求
    func peripheralManager(_ peripheral: CBPeripheralManager, didReceiveRead request: CBATTRequest) {
        print("收到讀取要求")
        var result: CBATTError.Code = .success
        switch request.characteristic.uuid.uuidString {
        case CharacteristicType.C001.rawValue:
            // 傳回最新時間
            let date = Date().description.data(using: .utf8)!
            request.value = date
            
        default:
            result = .requestNotSupported
        }
        
        peripheral.respond(to: request, withResult: result)
    }
    
    // MARK: - 收到central端寫入要求
    func peripheralManager(_ peripheral: CBPeripheralManager, didReceiveWrite requests: [CBATTRequest]) {
        print("收到寫入要求")
        guard let at = requests.first, let data = at.value else {
            return
        }
        
        switch at.characteristic.uuid.uuidString {
        case CharacteristicType.C001.rawValue:
            // 停止廣播，讓其他 central 搜尋不到本裝置
            // 然後不用回傳訊息給 central
            peripheral.stopAdvertising()
            
        case CharacteristicType.C002.rawValue:
            if let string = String(data: data, encoding: .utf8) {
                print(string)
            }
            // 需回傳訊息
            peripheral.respond(to: at, withResult: .success)
            
        default:
            break
        }
    }
}
