//
//  AVLibrary.swift
//  VisionDemo
//
//  Created by 朱克剛 on 2022/9/22.
//

import Foundation
import SwiftUI
import AVFoundation
import Vision

class FaceVideo: NSObject, ObservableObject {
    static let shared = FaceVideo()
    @Published var uiImage: UIImage?
    @Published var faces: [VNFaceObservation] = []
}

class CameraManager: NSObject, AVCaptureVideoDataOutputSampleBufferDelegate {
    static let current = CameraManager()
    let session = AVCaptureSession()
    // 前鏡頭
    private var frontCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .front
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        return nil
    }
    // 後鏡頭
    private var backCamera: AVCaptureDeviceInput? {
        let camera = AVCaptureDevice.DiscoverySession(
            deviceTypes: [.builtInWideAngleCamera],
            mediaType: .video,
            position: .back
        ).devices.first
        
        if let camera {
            return try? AVCaptureDeviceInput(device: camera)
        }
        return nil
    }
    
    private let videoOutput = AVCaptureVideoDataOutput()
    override init() {
        super.init()
        if let frontCamera {
            videoOutput.alwaysDiscardsLateVideoFrames = true
            videoOutput.setSampleBufferDelegate(self, queue: .global())
            
            session.addInput(frontCamera)
            session.addOutput(videoOutput)
            session.sessionPreset = .vga640x480
        }
    }
    
    func faceDetection(frame: CMSampleBuffer) {
        let request = VNDetectFaceLandmarksRequest { request, error in
            guard error == nil else {
                print(error!)
                return
            }
            
            if let faces = request.results as? [VNFaceObservation] {
                DispatchQueue.main.async {
                    FaceVideo.shared.faces = faces
                }
            }
        }
        
        do {
            let handler = VNImageRequestHandler(cmSampleBuffer: frame)
            try handler.perform([request])
        } catch {
            print(error)
        }
    }
    
    func sampleBufferToUIImage(_ frame: CMSampleBuffer) -> UIImage? {
        let cvBuffer = CMSampleBufferGetImageBuffer(frame)
        let ciImage = CIImage(cvImageBuffer: cvBuffer!)
        let context = CIContext()
        let imageRect = CGRect(
            x: 0,
            y: 0,
            width: CVPixelBufferGetWidth(cvBuffer!),
            height: CVPixelBufferGetHeight(cvBuffer!)
        )
        if let image = context.createCGImage(ciImage, from: imageRect) {
            let uiImage = UIImage(
                cgImage: image,
                scale: UIScreen.main.scale,
                orientation: .up
            )
            return uiImage
        } else {
            return nil
        }
    }
    
    func captureOutput(_ output: AVCaptureOutput, didOutput sampleBuffer: CMSampleBuffer, from connection: AVCaptureConnection) {
        connection.videoOrientation = .portrait
        // 將原始影像格式轉成UIImage格式
        if let uiImage = sampleBufferToUIImage(sampleBuffer) {
            DispatchQueue.main.async {
                FaceVideo.shared.uiImage = uiImage
            }
        }
        
        // 分析影像中的人臉
        faceDetection(frame: sampleBuffer)
    }
    
    func getPreviewLayer() -> CALayer {
        let layer = AVCaptureVideoPreviewLayer(session: session)
        layer.name = "preview"
        layer.videoGravity = .resizeAspectFill
        return layer
    }
}

struct PreviewView: UIViewRepresentable {
    var previewLayer: CALayer
    var proxy: GeometryProxy
    
    func makeUIView(context: Context) -> UIView {
        let view = UIView()
        view.layer.addSublayer(previewLayer)
        return view
    }
    
    func updateUIView(_ uiView: UIView, context: Context) {
        let previewLayer = uiView.layer.sublayers?.first { layer in
            layer.name == "preview"
        }
        if let previewLayer {
            previewLayer.frame = proxy.frame(in: .local)
        }
    }
    
    typealias UIViewType = UIView
}
