//
//  DataSourceIsVideoFile.swift
//  VisionDemo
//
//  Created by 朱克剛 on 2022/10/5.
//

import SwiftUI
import AVFoundation

class VideoSource: ObservableObject {
    @Published var uiImage: UIImage?
    
    private func load() async throws -> AVAssetReader {
        let url = Bundle.main.url(forResource: "IMG_4296.MOV", withExtension: nil)
        let asset = AVAsset(url: url!)
        let track = try await asset.loadTracks(withMediaType: .video)
        let output = AVAssetReaderTrackOutput(
            track: track[0],
            outputSettings: [
                String(kCVPixelBufferPixelFormatTypeKey):
                kCVPixelFormatType_420YpCbCr8BiPlanarFullRange
            ]
        )
        
        let reader = try AVAssetReader(asset: asset)
        reader.add(output)
        reader.startReading()
        
        return reader
    }
    
    func play() {
        Task {
            let reader = try await load()
            let waitTime = UInt64(1 / 30.0 * 1_000_000_000)
            var n = 1
            
            while reader.status == .reading {
                if let sampleBuffer = reader.outputs[0].copyNextSampleBuffer() {
                    DispatchQueue.main.async {
                        self.uiImage = self.sampleBufferToUIImage(sampleBuffer)
                    }
                    try await Task.sleep(nanoseconds: waitTime)
                    print("frame: \(n)")
                    n += 1
                    
                    // 這個位置進行影像分析處理，例如人臉辨識...等
                }
            }
        }
    }
    
    private func sampleBufferToUIImage(_ frame: CMSampleBuffer) -> UIImage? {
        let cvBuffer = CMSampleBufferGetImageBuffer(frame)
        let ciImage = CIImage(cvImageBuffer: cvBuffer!)
        let context = CIContext()
        let imageRect = CGRect(
            x: 0,
            y: 0,
            width: CVPixelBufferGetWidth(cvBuffer!),
            height: CVPixelBufferGetHeight(cvBuffer!)
        )
        
        if let image = context.createCGImage(ciImage, from: imageRect) {
            let uiImage = UIImage(
                cgImage: image,
                scale: UIScreen.main.scale,
                orientation: .up
            )
            return uiImage
        } else {
            return nil
        }
    }
}

struct DataSourceIsVideoFile: View {
    @ObservedObject private var video = VideoSource()
    private var ratio: Double? {
        if let size = video.uiImage?.size {
            return size.width / size.height
        }
        return nil
    }
    
    var body: some View {
        VStack {
            Button("Load video and play") {
                video.play()
            }
            Canvas { context, size in
                if let uiImage = video.uiImage {
                    context.draw(
                        Image(uiImage: uiImage),
                        in: CGRect(origin: .zero, size: size)
                    )
                }
            }
            .frame(width: 400, height: 400 / (ratio ?? 1))
        }
    }

}

struct DataSourceIsVideoFile_Previews: PreviewProvider {
    static var previews: some View {
        DataSourceIsVideoFile()
    }
}
