//
//  FaceDetect.swift
//  VisionDemo
//
//  Created by 朱克剛 on 2022/9/21.
//

import SwiftUI
import Vision

struct FaceDetect: View {
    private let uiImage = UIImage(named: "lena.jpg")!
    private var ratio: Double {
        uiImage.size.width / uiImage.size.height
    }
    @State private var faces: [VNFaceObservation] = []
    
    var body: some View {
        Canvas { context, size in
            // 畫出圖片
            context.draw(
                Image(uiImage: uiImage),
                in: CGRect(origin: .zero, size: size)
            )
            
            // 針對人臉進行處理
            faces.forEach { face in
                // 畫出範圍
                let box = face.boundingBox
                var rect = VNImageRectForNormalizedRect(
                    box, Int(size.width), Int(size.height)
                )
                rect = CGRect(
                    x: rect.origin.x,
                    y: size.height - rect.origin.y,
                    width: rect.width,
                    height: -rect.height
                )
                
                var path = Path()
                path.addRect(rect)
                context.stroke(path, with: .color(.yellow), lineWidth: 5)
                
                // 畫出特徵點
                face.landmarks?.nose?.pointsInImage(imageSize: size).forEach { point in
                    let center = CGPoint(
                        x: point.x,
                        y: size.height - point.y
                    )
                    var path = Path()
                    path.addArc(
                        center: center,
                        radius: 3,
                        startAngle: .zero,
                        endAngle: .degrees(360),
                        clockwise: true
                    )
                    context.fill(path, with: .color(.white))
                }
            }
        }
        .frame(width: 400, height: 400 / ratio)
        .onAppear() {
            Task {
                await faceDetection()
            }
        }
    }
    
    private func faceDetection() async {
        let request = VNDetectFaceLandmarksRequest { request, error in
            guard error == nil else {
                print(error!)
                return
            }
            
            if let faces = request.results as? [VNFaceObservation] {
                self.faces = faces
            }
        }
        
        do {
            let handler = VNImageRequestHandler(cgImage: uiImage.cgImage!)
            try handler.perform([request])
        } catch {
            print(error)
        }
    }
}

struct FaceDetect_Previews: PreviewProvider {
    static var previews: some View {
        FaceDetect()
    }
}
