<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2011 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: luofei614 <weibo.com/luofei614>　
// +----------------------------------------------------------------------
namespace Think;
/**
 * 權限認證類
 * 功能特性：
 * 1，是對規則進行認證，不是對節點進行認證。用戶可以把節點當作規則名稱實現對節點進行認證。
 *      $auth=new Auth();  $auth->check('規則名稱','用戶id')
 * 2，可以同時對多條規則進行認證，並設置多條規則的關係（or或者and）
 *      $auth=new Auth();  $auth->check('規則1,規則2','用戶id','and') 
 *      第三個參數為and時表示，用戶需要同時具有規則1和規則2的權限。 當第三個參數為or時，表示用戶值需要具備其中一個條件即可。默認為or
 * 3，一個用戶可以屬於多個用戶組(think_auth_group_access表 定義了用戶所屬用戶組)。我們需要設置每個用戶組擁有哪些規則(think_auth_group 定義了用戶組權限)
 * 
 * 4，支持規則表達式。
 *      在think_auth_rule 表中定義一條規則時，如果type為1， condition字段就可以定義規則表達式。 如定義{score}>5  and {score}<100  表示用戶的分數在5-100之間時這條規則才會通過。
 */

//數據庫
/*
-- ----------------------------
-- think_auth_rule，規則表，
-- id:主鍵，name：規則唯一標識, title：規則中文名稱 status 狀態：為1正常，為0禁用，condition：規則表達式，為空表示存在就驗證，不為空表示按照條件驗證
-- ----------------------------
 DROP TABLE IF EXISTS `think_auth_rule`;
CREATE TABLE `think_auth_rule` (  
    `id` mediumint(8) unsigned NOT NULL AUTO_INCREMENT,  
    `name` char(80) NOT NULL DEFAULT '',  
    `title` char(20) NOT NULL DEFAULT '',  
    `type` tinyint(1) NOT NULL DEFAULT '1',    
    `status` tinyint(1) NOT NULL DEFAULT '1',  
    `condition` char(100) NOT NULL DEFAULT '',  # 規則附件條件,滿足附加條件的規則,才認為是有效的規則
    PRIMARY KEY (`id`),  
    UNIQUE KEY `name` (`name`)
) ENGINE=MyISAM  DEFAULT CHARSET=utf8;
-- ----------------------------
-- think_auth_group 用戶組表， 
-- id：主鍵， title:用戶組中文名稱， rules：用戶組擁有的規則id， 多個規則","隔開，status 狀態：為1正常，為0禁用
-- ----------------------------
 DROP TABLE IF EXISTS `think_auth_group`;
CREATE TABLE `think_auth_group` ( 
    `id` mediumint(8) unsigned NOT NULL AUTO_INCREMENT, 
    `title` char(100) NOT NULL DEFAULT '', 
    `status` tinyint(1) NOT NULL DEFAULT '1', 
    `rules` char(80) NOT NULL DEFAULT '', 
    PRIMARY KEY (`id`)
) ENGINE=MyISAM  DEFAULT CHARSET=utf8;
-- ----------------------------
-- think_auth_group_access 用戶組明細表
-- uid:用戶id，group_id：用戶組id
-- ----------------------------
DROP TABLE IF EXISTS `think_auth_group_access`;
CREATE TABLE `think_auth_group_access` (  
    `uid` mediumint(8) unsigned NOT NULL,  
    `group_id` mediumint(8) unsigned NOT NULL, 
    UNIQUE KEY `uid_group_id` (`uid`,`group_id`),  
    KEY `uid` (`uid`), 
    KEY `group_id` (`group_id`)
) ENGINE=MyISAM DEFAULT CHARSET=utf8;
 */

class Auth{

    //默認配置
    protected $_config = array(
        'AUTH_ON'           => true,                      // 認證開關
        'AUTH_TYPE'         => 1,                         // 認證方式，1為實時認證；2為登錄認證。
        'AUTH_GROUP'        => 'auth_group',        // 用戶組數據表名
        'AUTH_GROUP_ACCESS' => 'auth_group_access', // 用戶-用戶組關係表
        'AUTH_RULE'         => 'auth_rule',         // 權限規則表
        'AUTH_USER'         => 'member'             // 用戶信息表
    );

    public function __construct() {
        $prefix = C('DB_PREFIX');
        $this->_config['AUTH_GROUP'] = $prefix.$this->_config['AUTH_GROUP'];
        $this->_config['AUTH_RULE'] = $prefix.$this->_config['AUTH_RULE'];
        $this->_config['AUTH_USER'] = $prefix.$this->_config['AUTH_USER'];
        $this->_config['AUTH_GROUP_ACCESS'] = $prefix.$this->_config['AUTH_GROUP_ACCESS'];
        if (C('AUTH_CONFIG')) {
            //可設置配置項 AUTH_CONFIG, 此配置項為數組。
            $this->_config = array_merge($this->_config, C('AUTH_CONFIG'));
        }
    }

    /**
      * 檢查權限
      * @param name string|array  需要驗證的規則列表,支持逗號分隔的權限規則或索引數組
      * @param uid  int           認證用戶的id
      * @param string mode        執行check的模式
      * @param relation string    如果為 'or' 表示滿足任一條規則即通過驗證;如果為 'and'則表示需滿足所有規則才能通過驗證
      * @return boolean           通過驗證返回true;失敗返回false
     */
    public function check($name, $uid, $type=1, $mode='url', $relation='or') {
        if (!$this->_config['AUTH_ON'])
            return true;
        $authList = $this->getAuthList($uid,$type); //獲取用戶需要驗證的所有有效規則列表
        if (is_string($name)) {
            $name = strtolower($name);
            if (strpos($name, ',') !== false) {
                $name = explode(',', $name);
            } else {
                $name = array($name);
            }
        }
        $list = array(); //保存驗證通過的規則名
        if ($mode=='url') {
            $REQUEST = unserialize( strtolower(serialize($_REQUEST)) );
        }
        foreach ( $authList as $auth ) {
            $query = preg_replace('/^.+\?/U','',$auth);
            if ($mode=='url' && $query!=$auth ) {
                parse_str($query,$param); //解析規則中的param
                $intersect = array_intersect_assoc($REQUEST,$param);
                $auth = preg_replace('/\?.*$/U','',$auth);
                if ( in_array($auth,$name) && $intersect==$param ) {  //如果節點相符且url參數滿足
                    $list[] = $auth ;
                }
            }else if (in_array($auth , $name)){
                $list[] = $auth ;
            }
        }
        if ($relation == 'or' and !empty($list)) {
            return true;
        }
        $diff = array_diff($name, $list);
        if ($relation == 'and' and empty($diff)) {
            return true;
        }
        return false;
    }

    /**
     * 根據用戶id獲取用戶組,返回值為數組
     * @param  uid int     用戶id
     * @return array       用戶所屬的用戶組 array(
     *     array('uid'=>'用戶id','group_id'=>'用戶組id','title'=>'用戶組名稱','rules'=>'用戶組擁有的規則id,多個,號隔開'),
     *     ...)   
     */
    public function getGroups($uid) {
        static $groups = array();
        if (isset($groups[$uid]))
            return $groups[$uid];
        $user_groups = M()
            ->table($this->_config['AUTH_GROUP_ACCESS'] . ' a')
            ->where("a.uid='$uid' and g.status='1'")
            ->join($this->_config['AUTH_GROUP']." g on a.group_id=g.id")
            ->field('uid,group_id,title,rules')->select();
        $groups[$uid]=$user_groups?:array();
        return $groups[$uid];
    }

    /**
     * 獲得權限列表
     * @param integer $uid  用戶id
     * @param integer $type 
     */
    protected function getAuthList($uid,$type) {
        static $_authList = array(); //保存用戶驗證通過的權限列表
        $t = implode(',',(array)$type);
        if (isset($_authList[$uid.$t])) {
            return $_authList[$uid.$t];
        }
        if( $this->_config['AUTH_TYPE']==2 && isset($_SESSION['_AUTH_LIST_'.$uid.$t])){
            return $_SESSION['_AUTH_LIST_'.$uid.$t];
        }

        //讀取用戶所屬用戶組
        $groups = $this->getGroups($uid);
        $ids = array();//保存用戶所屬用戶組設置的所有權限規則id
        foreach ($groups as $g) {
            $ids = array_merge($ids, explode(',', trim($g['rules'], ',')));
        }
        $ids = array_unique($ids);
        if (empty($ids)) {
            $_authList[$uid.$t] = array();
            return array();
        }

        $map=array(
            'id'=>array('in',$ids),
            'type'=>$type,
            'status'=>1,
        );
        //讀取用戶組所有權限規則
        $rules = M()->table($this->_config['AUTH_RULE'])->where($map)->field('condition,name')->select();

        //循環規則，判斷結果。
        $authList = array();   //
        foreach ($rules as $rule) {
            if (!empty($rule['condition'])) { //根據condition進行驗證
                $user = $this->getUserInfo($uid);//獲取用戶信息,一維數組

                $command = preg_replace('/\{(\w*?)\}/', '$user[\'\\1\']', $rule['condition']);
                //dump($command);//debug
                @(eval('$condition=(' . $command . ');'));
                if ($condition) {
                    $authList[] = strtolower($rule['name']);
                }
            } else {
                //只要存在就記錄
                $authList[] = strtolower($rule['name']);
            }
        }
        $_authList[$uid.$t] = $authList;
        if($this->_config['AUTH_TYPE']==2){
            //規則列表結果保存到session
            $_SESSION['_AUTH_LIST_'.$uid.$t]=$authList;
        }
        return array_unique($authList);
    }

    /**
     * 獲得用戶資料,根據自己的情況讀取數據庫
     */
    protected function getUserInfo($uid) {
        static $userinfo=array();
        if(!isset($userinfo[$uid])){
             $userinfo[$uid]=M()->where(array('uid'=>$uid))->table($this->_config['AUTH_USER'])->find();
        }
        return $userinfo[$uid];
    }

}
