<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2006-2014 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------

namespace Think\Db;
use Think\Config;
use Think\Debug;
use Think\Log;
use PDO;

class Lite {
    // PDO操作實例
    protected $PDOStatement = null;
    // 當前操作所屬的模型名
    protected $model      = '_think_';
    // 當前SQL指令
    protected $queryStr   = '';
    protected $modelSql   = array();
    // 最後插入ID
    protected $lastInsID  = null;
    // 返回或者影響記錄數
    protected $numRows    = 0;
    // 事務指令數
    protected $transTimes = 0;
    // 錯誤信息
    protected $error      = '';
    // 數據庫連接ID 支持多個連接
    protected $linkID     = array();
    // 當前連接ID
    protected $_linkID    = null;
    // 數據庫連接參數配置
    protected $config     = array(
        'type'              =>  '',     // 數據庫類型
        'hostname'          =>  '127.0.0.1', // 服務器地址
        'database'          =>  '',          // 數據庫名
        'username'          =>  '',      // 用戶名
        'password'          =>  '',          // 密碼
        'hostport'          =>  '',        // 端口     
        'dsn'               =>  '', //          
        'params'            =>  array(), // 數據庫連接參數        
        'charset'           =>  'utf8',      // 數據庫編碼默認採用utf8  
        'prefix'            =>  '',    // 數據庫表前綴
        'debug'             =>  false, // 數據庫調試模式
        'deploy'            =>  0, // 數據庫部署方式:0 集中式(單一服務器),1 分佈式(主從服務器)
        'rw_separate'       =>  false,       // 數據庫讀寫是否分離 主從式有效
        'master_num'        =>  1, // 讀寫分離後 主服務器數量
        'slave_no'          =>  '', // 指定從服務器序號
    );
    // 數據庫表達式
    protected $comparison = array('eq'=>'=','neq'=>'<>','gt'=>'>','egt'=>'>=','lt'=>'<','elt'=>'<=','notlike'=>'NOT LIKE','like'=>'LIKE','in'=>'IN','notin'=>'NOT IN');
    // 查詢表達式
    protected $selectSql  = 'SELECT%DISTINCT% %FIELD% FROM %TABLE%%JOIN%%WHERE%%GROUP%%HAVING%%ORDER%%LIMIT% %UNION%%COMMENT%';
    // 查詢次數
    protected $queryTimes   =   0;
    // 執行次數
    protected $executeTimes =   0;
    // PDO連接參數
    protected $options = array(
        PDO::ATTR_CASE              =>  PDO::CASE_LOWER,
        PDO::ATTR_ERRMODE           =>  PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_ORACLE_NULLS      =>  PDO::NULL_NATURAL,
        PDO::ATTR_STRINGIFY_FETCHES =>  false,
    );

    /**
     * 架構函數 讀取數據庫配置信息
     * @access public
     * @param array $config 數據庫配置數組
     */
    public function __construct($config=''){
        if(!empty($config)) {
            $this->config           =   array_merge($this->config,$config);
            if(is_array($this->config['params'])){
                $this->options  +=   $this->config['params'];
            }
        }
    }

    /**
     * 連接數據庫方法
     * @access public
     */
    public function connect($config='',$linkNum=0) {
        if ( !isset($this->linkID[$linkNum]) ) {
            if(empty($config))  $config =   $this->config;
            try{
                if(empty($config['dsn'])) {
                    $config['dsn']  =   $this->parseDsn($config);
                }
                if(version_compare(PHP_VERSION,'5.3.6','<=')){ //禁用模擬預處理語句
                    $this->options[PDO::ATTR_EMULATE_PREPARES]  =   false;
                }
                $this->linkID[$linkNum] = new PDO( $config['dsn'], $config['username'], $config['password'],$this->options);
            }catch (\PDOException $e) {
                E($e->getMessage());
            }
        }
        return $this->linkID[$linkNum];
    }

    /**
     * 解析pdo連接的dsn信息
     * @access public
     * @param array $config 連接信息
     * @return string
     */
    protected function parseDsn($config){}

    /**
     * 釋放查詢結果
     * @access public
     */
    public function free() {
        $this->PDOStatement = null;
    }

    /**
     * 執行查詢 返回數據集
     * @access public
     * @param string $str  sql指令
     * @param array $bind  參數綁定
     * @return mixed
     */
    public function query($str,$bind=array()) {
        $this->initConnect(false);
        if ( !$this->_linkID ) return false;
        $this->queryStr     =   $str;
        if(!empty($bind)){
            $that   =   $this;
            $this->queryStr =   strtr($this->queryStr,array_map(function($val) use($that){ return '\''.$that->escapeString($val).'\''; },$bind));
        }
        //釋放前次的查詢結果
        if ( !empty($this->PDOStatement) ) $this->free();
        $this->queryTimes++;
        N('db_query',1); // 兼容代碼        
        // 調試開始
        $this->debug(true);
        $this->PDOStatement = $this->_linkID->prepare($str);
        if(false === $this->PDOStatement)
            E($this->error());
        foreach ($bind as $key => $val) {
            if(is_array($val)){
                $this->PDOStatement->bindValue($key, $val[0], $val[1]);
            }else{
                $this->PDOStatement->bindValue($key, $val);
            }
        }
        $result =   $this->PDOStatement->execute();
        // 調試結束
        $this->debug(false);
        if ( false === $result ) {
            $this->error();
            return false;
        } else {
            return $this->getResult();
        }
    }

    /**
     * 執行語句
     * @access public
     * @param string $str  sql指令
     * @param array $bind  參數綁定
     * @return integer
     */
    public function execute($str,$bind=array()) {
        $this->initConnect(true);
        if ( !$this->_linkID ) return false;
        $this->queryStr = $str;
        if(!empty($bind)){
            $that   =   $this;
            $this->queryStr =   strtr($this->queryStr,array_map(function($val) use($that){ return '\''.$that->escapeString($val).'\''; },$bind));
        }      
        //釋放前次的查詢結果
        if ( !empty($this->PDOStatement) ) $this->free();
        $this->executeTimes++;
        N('db_write',1); // 兼容代碼        
        // 記錄開始執行時間
        $this->debug(true);
        $this->PDOStatement =   $this->_linkID->prepare($str);
        if(false === $this->PDOStatement) {
            E($this->error());
        }
        foreach ($bind as $key => $val) {
            if(is_array($val)){
                $this->PDOStatement->bindValue($key, $val[0], $val[1]);
            }else{
                $this->PDOStatement->bindValue($key, $val);
            }
        }
        $result =   $this->PDOStatement->execute();
        $this->debug(false);
        if ( false === $result) {
            $this->error();
            return false;
        } else {
            $this->numRows = $this->PDOStatement->rowCount();
            if(preg_match("/^\s*(INSERT\s+INTO|REPLACE\s+INTO)\s+/i", $str)) {
                $this->lastInsID = $this->_linkID->lastInsertId();
            }
            return $this->numRows;
        }
    }

    /**
     * 啟動事務
     * @access public
     * @return void
     */
    public function startTrans() {
        $this->initConnect(true);
        if ( !$this->_linkID ) return false;
        //數據rollback 支持
        if ($this->transTimes == 0) {
            $this->_linkID->beginTransaction();
        }
        $this->transTimes++;
        return ;
    }

    /**
     * 用於非自動提交狀態下面的查詢提交
     * @access public
     * @return boolean
     */
    public function commit() {
        if ($this->transTimes > 0) {
            $result = $this->_linkID->commit();
            $this->transTimes = 0;
            if(!$result){
                $this->error();
                return false;
            }
        }
        return true;
    }

    /**
     * 事務回滾
     * @access public
     * @return boolean
     */
    public function rollback() {
        if ($this->transTimes > 0) {
            $result = $this->_linkID->rollback();
            $this->transTimes = 0;
            if(!$result){
                $this->error();
                return false;
            }
        }
        return true;
    }

    /**
     * 獲得所有的查詢數據
     * @access private
     * @return array
     */
    private function getResult() {
        //返回數據集
        $result =   $this->PDOStatement->fetchAll(PDO::FETCH_ASSOC);
        $this->numRows = count( $result );
        return $result;
    }

    /**
     * 獲得查詢次數
     * @access public
     * @param boolean $execute 是否包含所有查詢
     * @return integer
     */
    public function getQueryTimes($execute=false){
        return $execute?$this->queryTimes+$this->executeTimes:$this->queryTimes;
    }

    /**
     * 獲得執行次數
     * @access public
     * @return integer
     */
    public function getExecuteTimes(){
        return $this->executeTimes;
    }

    /**
     * 關閉數據庫
     * @access public
     */
    public function close() {
        $this->_linkID = null;
    }

    /**
     * 數據庫錯誤信息
     * 並顯示當前的SQL語句
     * @access public
     * @return string
     */
    public function error() {
        if($this->PDOStatement) {
            $error = $this->PDOStatement->errorInfo();
            $this->error = $error[1].':'.$error[2];
        }else{
            $this->error = '';
        }
        if('' != $this->queryStr){
            $this->error .= "\n [ SQL語句 ] : ".$this->queryStr;
        }
        // 記錄錯誤日誌
        trace($this->error,'','ERR');
        if($this->config['debug']) {// 開啟數據庫調試模式
            E($this->error);
        }else{
            return $this->error;
        }
    }

    /**
     * 獲取最近一次查詢的sql語句 
     * @param string $model  模型名
     * @access public
     * @return string
     */
    public function getLastSql($model='') {
        return $model?$this->modelSql[$model]:$this->queryStr;
    }

    /**
     * 獲取最近插入的ID
     * @access public
     * @return string
     */
    public function getLastInsID() {
        return $this->lastInsID;
    }

    /**
     * 獲取最近的錯誤信息
     * @access public
     * @return string
     */
    public function getError() {
        return $this->error;
    }

    /**
     * SQL指令安全過濾
     * @access public
     * @param string $str  SQL字符串
     * @return string
     */
    public function escapeString($str) {
        return addslashes($str);
    }

    /**
     * 設置當前操作模型
     * @access public
     * @param string $model  模型名
     * @return void
     */
    public function setModel($model){
        $this->model =  $model;
    }

    /**
     * 數據庫調試 記錄當前SQL
     * @access protected
     * @param boolean $start  調試開始標記 true 開始 false 結束
     */
    protected function debug($start) {
        if($this->config['debug']) {// 開啟數據庫調試模式
            if($start) {
                G('queryStartTime');
            }else{
                $this->modelSql[$this->model]   =  $this->queryStr;
                //$this->model  =   '_think_';
                // 記錄操作結束時間
                G('queryEndTime');
                trace($this->queryStr.' [ RunTime:'.G('queryStartTime','queryEndTime').'s ]','','SQL');
            }
        }
    }

    /**
     * 初始化數據庫連接
     * @access protected
     * @param boolean $master 主服務器
     * @return void
     */
    protected function initConnect($master=true) {
        if(!empty($this->config['deploy']))
            // 採用分佈式數據庫
            $this->_linkID = $this->multiConnect($master);
        else
            // 默認單數據庫
            if ( !$this->_linkID ) $this->_linkID = $this->connect();
    }

    /**
     * 連接分佈式服務器
     * @access protected
     * @param boolean $master 主服務器
     * @return void
     */
    protected function multiConnect($master=false) {
        // 分佈式數據庫配置解析
        $_config['username']    =   explode(',',$this->config['username']);
        $_config['password']    =   explode(',',$this->config['password']);
        $_config['hostname']    =   explode(',',$this->config['hostname']);
        $_config['hostport']    =   explode(',',$this->config['hostport']);
        $_config['database']    =   explode(',',$this->config['database']);
        $_config['dsn']         =   explode(',',$this->config['dsn']);
        $_config['charset']     =   explode(',',$this->config['charset']);

        // 數據庫讀寫是否分離
        if($this->config['rw_separate']){
            // 主從式採用讀寫分離
            if($master)
                // 主服務器寫入
                $r  =   floor(mt_rand(0,$this->config['master_num']-1));
            else{
                if(is_numeric($this->config['slave_no'])) {// 指定服務器讀
                    $r = $this->config['slave_no'];
                }else{
                    // 讀操作連接從服務器
                    $r = floor(mt_rand($this->config['master_num'],count($_config['hostname'])-1));   // 每次隨機連接的數據庫
                }
            }
        }else{
            // 讀寫操作不區分服務器
            $r = floor(mt_rand(0,count($_config['hostname'])-1));   // 每次隨機連接的數據庫
        }
        $db_config = array(
            'username'  =>  isset($_config['username'][$r])?$_config['username'][$r]:$_config['username'][0],
            'password'  =>  isset($_config['password'][$r])?$_config['password'][$r]:$_config['password'][0],
            'hostname'  =>  isset($_config['hostname'][$r])?$_config['hostname'][$r]:$_config['hostname'][0],
            'hostport'  =>  isset($_config['hostport'][$r])?$_config['hostport'][$r]:$_config['hostport'][0],
            'database'  =>  isset($_config['database'][$r])?$_config['database'][$r]:$_config['database'][0],
            'dsn'       =>  isset($_config['dsn'][$r])?$_config['dsn'][$r]:$_config['dsn'][0],
            'charset'   =>  isset($_config['charset'][$r])?$_config['charset'][$r]:$_config['charset'][0],
        );
        return $this->connect($db_config,$r);
    }

   /**
     * 析構方法
     * @access public
     */
    public function __destruct() {
        // 釋放查詢
        if ($this->PDOStatement){
            $this->free();
        }
        // 關閉連接
        $this->close();
    }
}