<?php
// +----------------------------------------------------------------------
// | TOPThink [ WE CAN DO IT JUST THINK ]
// +----------------------------------------------------------------------
// | Copyright (c) 2010 http://topthink.com All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: 麥當苗兒 <zuojiazi@vip.qq.com> <http://www.zjzit.cn>
// +----------------------------------------------------------------------
// | ThinkImage.class.php 2013-03-05
// +----------------------------------------------------------------------

namespace Think;

/**
 * 圖片處理驅動類，可配置圖片處理庫
 * 目前支持GD庫和imagick
 * @author 麥當苗兒 <zuojiazi@vip.qq.com>
 */
class Image{
    /* 驅動相關常量定義 */
    const IMAGE_GD              =   1; //常量，標識GD庫類型
    const IMAGE_IMAGICK         =   2; //常量，標識imagick庫類型

    /* 縮略圖相關常量定義 */
    const IMAGE_THUMB_SCALE     =   1 ; //常量，標識縮略圖等比例縮放類型
    const IMAGE_THUMB_FILLED    =   2 ; //常量，標識縮略圖縮放後填充類型
    const IMAGE_THUMB_CENTER    =   3 ; //常量，標識縮略圖居中裁剪類型
    const IMAGE_THUMB_NORTHWEST =   4 ; //常量，標識縮略圖左上角裁剪類型
    const IMAGE_THUMB_SOUTHEAST =   5 ; //常量，標識縮略圖右下角裁剪類型
    const IMAGE_THUMB_FIXED     =   6 ; //常量，標識縮略圖固定尺寸縮放類型

    /* 水印相關常量定義 */
    const IMAGE_WATER_NORTHWEST =   1 ; //常量，標識左上角水印
    const IMAGE_WATER_NORTH     =   2 ; //常量，標識上居中水印
    const IMAGE_WATER_NORTHEAST =   3 ; //常量，標識右上角水印
    const IMAGE_WATER_WEST      =   4 ; //常量，標識左居中水印
    const IMAGE_WATER_CENTER    =   5 ; //常量，標識居中水印
    const IMAGE_WATER_EAST      =   6 ; //常量，標識右居中水印
    const IMAGE_WATER_SOUTHWEST =   7 ; //常量，標識左下角水印
    const IMAGE_WATER_SOUTH     =   8 ; //常量，標識下居中水印
    const IMAGE_WATER_SOUTHEAST =   9 ; //常量，標識右下角水印

    /**
     * 圖片資源
     * @var resource
     */
    private $img;

    /**
     * 構造方法，用於實例化一個圖片處理對像
     * @param string $type 要使用的類庫，默認使用GD庫
     */
    public function __construct($type = self::IMAGE_GD, $imgname = null){
        /* 判斷調用庫的類型 */
        switch ($type) {
            case self::IMAGE_GD:
                $class = 'Gd';
                break;
            case self::IMAGE_IMAGICK:
                $class = 'Imagick';
                break;
            default:
                E('不支持的圖片處理庫類型');
        }

        /* 引入處理庫，實例化圖片處理對像 */
        $class  =    "Think\\Image\\Driver\\{$class}";
        $this->img = new $class($imgname);
    }

    /**
     * 打開一幅圖像
     * @param  string $imgname 圖片路徑
     * @return Object          當前圖片處理庫對像
     */
    public function open($imgname){
        $this->img->open($imgname);
        return $this;
    }

    /**
     * 保存圖片
     * @param  string  $imgname   圖片保存名稱
     * @param  string  $type      圖片類型
     * @param  integer $quality   圖像質量      
     * @param  boolean $interlace 是否對JPEG類型圖片設置隔行掃瞄
     * @return Object             當前圖片處理庫對像
     */
    public function save($imgname, $type = null, $quality=80,$interlace = true){
        $this->img->save($imgname, $type, $quality,$interlace);
        return $this;
    }

    /**
     * 返回圖片寬度
     * @return integer 圖片寬度
     */
    public function width(){
        return $this->img->width();
    }

    /**
     * 返回圖片高度
     * @return integer 圖片高度
     */
    public function height(){
        return $this->img->height();
    }

    /**
     * 返回圖像類型
     * @return string 圖片類型
     */
    public function type(){
        return $this->img->type();
    }

    /**
     * 返回圖像MIME類型
     * @return string 圖像MIME類型
     */
    public function mime(){
        return $this->img->mime();
    }

    /**
     * 返回圖像尺寸數組 0 - 圖片寬度，1 - 圖片高度
     * @return array 圖片尺寸
     */
    public function size(){
        return $this->img->size();
    }

    /**
     * 裁剪圖片
     * @param  integer $w      裁剪區域寬度
     * @param  integer $h      裁剪區域高度
     * @param  integer $x      裁剪區域x坐標
     * @param  integer $y      裁剪區域y坐標
     * @param  integer $width  圖片保存寬度
     * @param  integer $height 圖片保存高度
     * @return Object          當前圖片處理庫對像
     */
    public function crop($w, $h, $x = 0, $y = 0, $width = null, $height = null){
        $this->img->crop($w, $h, $x, $y, $width, $height);
        return $this;
    }

    /**
     * 生成縮略圖
     * @param  integer $width  縮略圖最大寬度
     * @param  integer $height 縮略圖最大高度
     * @param  integer $type   縮略圖裁剪類型
     * @return Object          當前圖片處理庫對像
     */
    public function thumb($width, $height, $type = self::IMAGE_THUMB_SCALE){
        $this->img->thumb($width, $height, $type);
        return $this;
    }

    /**
     * 添加水印
     * @param  string  $source 水印圖片路徑
     * @param  integer $locate 水印位置
     * @param  integer $alpha  水印透明度
     * @return Object          當前圖片處理庫對像
     */
    public function water($source, $locate = self::IMAGE_WATER_SOUTHEAST,$alpha=80){
        $this->img->water($source, $locate,$alpha);
        return $this;
    }

    /**
     * 圖像添加文字
     * @param  string  $text   添加的文字
     * @param  string  $font   字體路徑
     * @param  integer $size   字號
     * @param  string  $color  文字顏色
     * @param  integer $locate 文字寫入位置
     * @param  integer $offset 文字相對當前位置的偏移量
     * @param  integer $angle  文字傾斜角度
     * @return Object          當前圖片處理庫對像
     */
    public function text($text, $font, $size, $color = '#00000000', 
        $locate = self::IMAGE_WATER_SOUTHEAST, $offset = 0, $angle = 0){
        $this->img->text($text, $font, $size, $color, $locate, $offset, $angle);
        return $this;
    }
}