<?php
// +----------------------------------------------------------------------
// | TOPThink [ WE CAN DO IT JUST THINK ]
// +----------------------------------------------------------------------
// | Copyright (c) 2010 http://topthink.com All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: 麥當苗兒 <zuojiazi@vip.qq.com> <http://www.zjzit.cn>
// +----------------------------------------------------------------------
// | ImageGd.class.php 2013-03-05
// +----------------------------------------------------------------------
namespace Think\Image\Driver;
use Think\Image;
class Gd{
    /**
     * 圖像資源對像
     * @var resource
     */
    private $img;

    /**
     * 圖像信息，包括width,height,type,mime,size
     * @var array
     */
    private $info;

    /**
     * 構造方法，可用於打開一張圖像
     * @param string $imgname 圖像路徑
     */
    public function __construct($imgname = null) {
        $imgname && $this->open($imgname);
    }

    /**
     * 打開一張圖像
     * @param  string $imgname 圖像路徑
     */
    public function open($imgname){
        //檢測圖像文件
        if(!is_file($imgname)) E('不存在的圖像文件');

        //獲取圖像信息
        $info = getimagesize($imgname);

        //檢測圖像合法性
        if(false === $info || (IMAGETYPE_GIF === $info[2] && empty($info['bits']))){
            E('非法圖像文件');
        }

        //設置圖像信息
        $this->info = array(
            'width'  => $info[0],
            'height' => $info[1],
            'type'   => image_type_to_extension($info[2], false),
            'mime'   => $info['mime'],
        );

        //銷毀已存在的圖像
        empty($this->img) || imagedestroy($this->img);

        //打開圖像
        if('gif' == $this->info['type']){
            $class  =    'Think\\Image\\Driver\\GIF';
            $this->gif = new $class($imgname);
            $this->img = imagecreatefromstring($this->gif->image());
        } else {
            $fun = "imagecreatefrom{$this->info['type']}";
            $this->img = $fun($imgname);
        }
    }

    /**
     * 保存圖像
     * @param  string  $imgname   圖像保存名稱
     * @param  string  $type      圖像類型
     * @param  integer $quality   圖像質量     
     * @param  boolean $interlace 是否對JPEG類型圖像設置隔行掃瞄
     */
    public function save($imgname, $type = null, $quality=80,$interlace = true){
        if(empty($this->img)) E('沒有可以被保存的圖像資源');

        //自動獲取圖像類型
        if(is_null($type)){
            $type = $this->info['type'];
        } else {
            $type = strtolower($type);
        }
        //保存圖像
        if('jpeg' == $type || 'jpg' == $type){
            //JPEG圖像設置隔行掃瞄
            imageinterlace($this->img, $interlace);
            imagejpeg($this->img, $imgname,$quality);
        }elseif('gif' == $type && !empty($this->gif)){
            $this->gif->save($imgname);
        }else{
            $fun  =   'image'.$type;
            $fun($this->img, $imgname);
        }
    }

    /**
     * 返回圖像寬度
     * @return integer 圖像寬度
     */
    public function width(){
        if(empty($this->img)) E('沒有指定圖像資源');
        return $this->info['width'];
    }

    /**
     * 返回圖像高度
     * @return integer 圖像高度
     */
    public function height(){
        if(empty($this->img)) E('沒有指定圖像資源');
        return $this->info['height'];
    }

    /**
     * 返回圖像類型
     * @return string 圖像類型
     */
    public function type(){
        if(empty($this->img)) E('沒有指定圖像資源');
        return $this->info['type'];
    }

    /**
     * 返回圖像MIME類型
     * @return string 圖像MIME類型
     */
    public function mime(){
        if(empty($this->img)) E('沒有指定圖像資源');
        return $this->info['mime'];
    }

    /**
     * 返回圖像尺寸數組 0 - 圖像寬度，1 - 圖像高度
     * @return array 圖像尺寸
     */
    public function size(){
        if(empty($this->img)) E('沒有指定圖像資源');
        return array($this->info['width'], $this->info['height']);
    }

    /**
     * 裁剪圖像
     * @param  integer $w      裁剪區域寬度
     * @param  integer $h      裁剪區域高度
     * @param  integer $x      裁剪區域x坐標
     * @param  integer $y      裁剪區域y坐標
     * @param  integer $width  圖像保存寬度
     * @param  integer $height 圖像保存高度
     */
    public function crop($w, $h, $x = 0, $y = 0, $width = null, $height = null){
        if(empty($this->img)) E('沒有可以被裁剪的圖像資源');

        //設置保存尺寸
        empty($width)  && $width  = $w;
        empty($height) && $height = $h;

        do {
            //創建新圖像
            $img = imagecreatetruecolor($width, $height);
            // 調整默認顏色
            $color = imagecolorallocate($img, 255, 255, 255);
            imagefill($img, 0, 0, $color);

            //裁剪
            imagecopyresampled($img, $this->img, 0, 0, $x, $y, $width, $height, $w, $h);
            imagedestroy($this->img); //銷毀原圖

            //設置新圖像
            $this->img = $img;
        } while(!empty($this->gif) && $this->gifNext());

        $this->info['width']  = $width;
        $this->info['height'] = $height;
    }

    /**
     * 生成縮略圖
     * @param  integer $width  縮略圖最大寬度
     * @param  integer $height 縮略圖最大高度
     * @param  integer $type   縮略圖裁剪類型
     */
    public function thumb($width, $height, $type = Image::IMAGE_THUMB_SCALE){
        if(empty($this->img)) E('沒有可以被縮略的圖像資源');

        //原圖寬度和高度
        $w = $this->info['width'];
        $h = $this->info['height'];

        /* 計算縮略圖生成的必要參數 */
        switch ($type) {
            /* 等比例縮放 */
            case Image::IMAGE_THUMB_SCALE:
                //原圖尺寸小於縮略圖尺寸則不進行縮略
                if($w < $width && $h < $height) return;

                //計算縮放比例
                $scale = min($width/$w, $height/$h);
                
                //設置縮略圖的坐標及寬度和高度
                $x = $y = 0;
                $width  = $w * $scale;
                $height = $h * $scale;
                break;

            /* 居中裁剪 */
            case Image::IMAGE_THUMB_CENTER:
                //計算縮放比例
                $scale = max($width/$w, $height/$h);

                //設置縮略圖的坐標及寬度和高度
                $w = $width/$scale;
                $h = $height/$scale;
                $x = ($this->info['width'] - $w)/2;
                $y = ($this->info['height'] - $h)/2;
                break;

            /* 左上角裁剪 */
            case Image::IMAGE_THUMB_NORTHWEST:
                //計算縮放比例
                $scale = max($width/$w, $height/$h);

                //設置縮略圖的坐標及寬度和高度
                $x = $y = 0;
                $w = $width/$scale;
                $h = $height/$scale;
                break;

            /* 右下角裁剪 */
            case Image::IMAGE_THUMB_SOUTHEAST:
                //計算縮放比例
                $scale = max($width/$w, $height/$h);

                //設置縮略圖的坐標及寬度和高度
                $w = $width/$scale;
                $h = $height/$scale;
                $x = $this->info['width'] - $w;
                $y = $this->info['height'] - $h;
                break;

            /* 填充 */
            case Image::IMAGE_THUMB_FILLED:
                //計算縮放比例
                if($w < $width && $h < $height){
                    $scale = 1;
                } else {
                    $scale = min($width/$w, $height/$h);
                }

                //設置縮略圖的坐標及寬度和高度
                $neww = $w * $scale;
                $newh = $h * $scale;
                $posx = ($width  - $w * $scale)/2;
                $posy = ($height - $h * $scale)/2;

                do{
                    //創建新圖像
                    $img = imagecreatetruecolor($width, $height);
                    // 調整默認顏色
                    $color = imagecolorallocate($img, 255, 255, 255);
                    imagefill($img, 0, 0, $color);

                    //裁剪
                    imagecopyresampled($img, $this->img, $posx, $posy, $x, $y, $neww, $newh, $w, $h);
                    imagedestroy($this->img); //銷毀原圖
                    $this->img = $img;
                } while(!empty($this->gif) && $this->gifNext());
                
                $this->info['width']  = $width;
                $this->info['height'] = $height;
                return;

            /* 固定 */
            case Image::IMAGE_THUMB_FIXED:
                $x = $y = 0;
                break;

            default:
                E('不支持的縮略圖裁剪類型');
        }

        /* 裁剪圖像 */
        $this->crop($w, $h, $x, $y, $width, $height);
    }

    /**
     * 添加水印
     * @param  string  $source 水印圖片路徑
     * @param  integer $locate 水印位置
     * @param  integer $alpha  水印透明度
     */
    public function water($source, $locate = Image::IMAGE_WATER_SOUTHEAST,$alpha=80){
        //資源檢測
        if(empty($this->img)) E('沒有可以被添加水印的圖像資源');
        if(!is_file($source)) E('水印圖像不存在');

        //獲取水印圖像信息
        $info = getimagesize($source);
        if(false === $info || (IMAGETYPE_GIF === $info[2] && empty($info['bits']))){
            E('非法水印文件');
        }

        //創建水印圖像資源
        $fun   = 'imagecreatefrom' . image_type_to_extension($info[2], false);
        $water = $fun($source);

        //設定水印圖像的混色模式
        imagealphablending($water, true);

        /* 設定水印位置 */
        switch ($locate) {
            /* 右下角水印 */
            case Image::IMAGE_WATER_SOUTHEAST:
                $x = $this->info['width'] - $info[0];
                $y = $this->info['height'] - $info[1];
                break;

            /* 左下角水印 */
            case Image::IMAGE_WATER_SOUTHWEST:
                $x = 0;
                $y = $this->info['height'] - $info[1];
                break;

            /* 左上角水印 */
            case Image::IMAGE_WATER_NORTHWEST:
                $x = $y = 0;
                break;

            /* 右上角水印 */
            case Image::IMAGE_WATER_NORTHEAST:
                $x = $this->info['width'] - $info[0];
                $y = 0;
                break;

            /* 居中水印 */
            case Image::IMAGE_WATER_CENTER:
                $x = ($this->info['width'] - $info[0])/2;
                $y = ($this->info['height'] - $info[1])/2;
                break;

            /* 下居中水印 */
            case Image::IMAGE_WATER_SOUTH:
                $x = ($this->info['width'] - $info[0])/2;
                $y = $this->info['height'] - $info[1];
                break;

            /* 右居中水印 */
            case Image::IMAGE_WATER_EAST:
                $x = $this->info['width'] - $info[0];
                $y = ($this->info['height'] - $info[1])/2;
                break;

            /* 上居中水印 */
            case Image::IMAGE_WATER_NORTH:
                $x = ($this->info['width'] - $info[0])/2;
                $y = 0;
                break;

            /* 左居中水印 */
            case Image::IMAGE_WATER_WEST:
                $x = 0;
                $y = ($this->info['height'] - $info[1])/2;
                break;

            default:
                /* 自定義水印坐標 */
                if(is_array($locate)){
                    list($x, $y) = $locate;
                } else {
                    E('不支持的水印位置類型');
                }
        }

        do{
            //添加水印
            $src = imagecreatetruecolor($info[0], $info[1]);
            // 調整默認顏色
            $color = imagecolorallocate($src, 255, 255, 255);
            imagefill($src, 0, 0, $color);

            imagecopy($src, $this->img, 0, 0, $x, $y, $info[0], $info[1]);
            imagecopy($src, $water, 0, 0, 0, 0, $info[0], $info[1]);
            imagecopymerge($this->img, $src, $x, $y, 0, 0, $info[0], $info[1], $alpha);

            //銷毀零時圖片資源
            imagedestroy($src);
        } while(!empty($this->gif) && $this->gifNext());

        //銷毀水印資源
        imagedestroy($water);
    }

    /**
     * 圖像添加文字
     * @param  string  $text   添加的文字
     * @param  string  $font   字體路徑
     * @param  integer $size   字號
     * @param  string  $color  文字顏色
     * @param  integer $locate 文字寫入位置
     * @param  integer $offset 文字相對當前位置的偏移量
     * @param  integer $angle  文字傾斜角度
     */
    public function text($text, $font, $size, $color = '#00000000', 
        $locate = Image::IMAGE_WATER_SOUTHEAST, $offset = 0, $angle = 0){
        //資源檢測
        if(empty($this->img)) E('沒有可以被寫入文字的圖像資源');
        if(!is_file($font)) E("不存在的字體文件：{$font}");

        //獲取文字信息
        $info = imagettfbbox($size, $angle, $font, $text);
        $minx = min($info[0], $info[2], $info[4], $info[6]); 
        $maxx = max($info[0], $info[2], $info[4], $info[6]); 
        $miny = min($info[1], $info[3], $info[5], $info[7]); 
        $maxy = max($info[1], $info[3], $info[5], $info[7]); 

        /* 計算文字初始坐標和尺寸 */
        $x = $minx;
        $y = abs($miny);
        $w = $maxx - $minx;
        $h = $maxy - $miny;

        /* 設定文字位置 */
        switch ($locate) {
            /* 右下角文字 */
            case Image::IMAGE_WATER_SOUTHEAST:
                $x += $this->info['width']  - $w;
                $y += $this->info['height'] - $h;
                break;

            /* 左下角文字 */
            case Image::IMAGE_WATER_SOUTHWEST:
                $y += $this->info['height'] - $h;
                break;

            /* 左上角文字 */
            case Image::IMAGE_WATER_NORTHWEST:
                // 起始坐標即為左上角坐標，無需調整
                break;

            /* 右上角文字 */
            case Image::IMAGE_WATER_NORTHEAST:
                $x += $this->info['width'] - $w;
                break;

            /* 居中文字 */
            case Image::IMAGE_WATER_CENTER:
                $x += ($this->info['width']  - $w)/2;
                $y += ($this->info['height'] - $h)/2;
                break;

            /* 下居中文字 */
            case Image::IMAGE_WATER_SOUTH:
                $x += ($this->info['width'] - $w)/2;
                $y += $this->info['height'] - $h;
                break;

            /* 右居中文字 */
            case Image::IMAGE_WATER_EAST:
                $x += $this->info['width'] - $w;
                $y += ($this->info['height'] - $h)/2;
                break;

            /* 上居中文字 */
            case Image::IMAGE_WATER_NORTH:
                $x += ($this->info['width'] - $w)/2;
                break;

            /* 左居中文字 */
            case Image::IMAGE_WATER_WEST:
                $y += ($this->info['height'] - $h)/2;
                break;

            default:
                /* 自定義文字坐標 */
                if(is_array($locate)){
                    list($posx, $posy) = $locate;
                    $x += $posx;
                    $y += $posy;
                } else {
                    E('不支持的文字位置類型');
                }
        }

        /* 設置偏移量 */
        if(is_array($offset)){
            $offset = array_map('intval', $offset);
            list($ox, $oy) = $offset;
        } else{
            $offset = intval($offset);
            $ox = $oy = $offset;
        }

        /* 設置顏色 */
        if(is_string($color) && 0 === strpos($color, '#')){
            $color = str_split(substr($color, 1), 2);
            $color = array_map('hexdec', $color);
            if(empty($color[3]) || $color[3] > 127){
                $color[3] = 0;
            }
        } elseif (!is_array($color)) {
            E('錯誤的顏色值');
        }

        do{
            /* 寫入文字 */
            $col = imagecolorallocatealpha($this->img, $color[0], $color[1], $color[2], $color[3]);
            imagettftext($this->img, $size, $angle, $x + $ox, $y + $oy, $col, $font, $text);
        } while(!empty($this->gif) && $this->gifNext());
    }

    /* 切換到GIF的下一幀並保存當前幀，內部使用 */
    private function gifNext(){
        ob_start();
        ob_implicit_flush(0);
        imagegif($this->img);
        $img = ob_get_clean();

        $this->gif->image($img);
        $next = $this->gif->nextImage();

        if($next){
            $this->img = imagecreatefromstring($next);
            return $next;
        } else {
            $this->img = imagecreatefromstring($this->gif->image());
            return false;
        }
    }

    /**
     * 析構方法，用於銷毀圖像資源
     */
    public function __destruct() {
        empty($this->img) || imagedestroy($this->img);
    }
}