<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2006-2014 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------
namespace Think;
/**
 * ThinkPHP Model模型類
 * 實現了ORM和ActiveRecords模式
 */
class Model {
    // 操作狀態
    const MODEL_INSERT          =   1;      //  插入模型數據
    const MODEL_UPDATE          =   2;      //  更新模型數據
    const MODEL_BOTH            =   3;      //  包含上面兩種方式
    const MUST_VALIDATE         =   1;      // 必須驗證
    const EXISTS_VALIDATE       =   0;      // 表單存在字段則驗證
    const VALUE_VALIDATE        =   2;      // 表單值不為空則驗證

    // 當前數據庫操作對像
    protected $db               =   null;
	// 數據庫對像池
	private   $_db				=	array();
    // 主鍵名稱
    protected $pk               =   'id';
    // 主鍵是否自動增長
    protected $autoinc          =   false;    
    // 數據表前綴
    protected $tablePrefix      =   null;
    // 模型名稱
    protected $name             =   '';
    // 數據庫名稱
    protected $dbName           =   '';
    //數據庫配置
    protected $connection       =   '';
    // 數據表名（不包含表前綴）
    protected $tableName        =   '';
    // 實際數據表名（包含表前綴）
    protected $trueTableName    =   '';
    // 最近錯誤信息
    protected $error            =   '';
    // 字段信息
    protected $fields           =   array();
    // 數據信息
    protected $data             =   array();
    // 查詢表達式參數
    protected $options          =   array();
    protected $_validate        =   array();  // 自動驗證定義
    protected $_auto            =   array();  // 自動完成定義
    protected $_map             =   array();  // 字段映射定義
    protected $_scope           =   array();  // 命名範圍定義
    // 是否自動檢測數據表字段信息
    protected $autoCheckFields  =   true;
    // 是否批處理驗證
    protected $patchValidate    =   false;
    // 鏈操作方法列表
    protected $methods          =   array('strict','order','alias','having','group','lock','distinct','auto','filter','validate','result','token','index','force');

    /**
     * 架構函數
     * 取得DB類的實例對像 字段檢查
     * @access public
     * @param string $name 模型名稱
     * @param string $tablePrefix 表前綴
     * @param mixed $connection 數據庫連接信息
     */
    public function __construct($name='',$tablePrefix='',$connection='') {
        // 模型初始化
        $this->_initialize();
        // 獲取模型名稱
        if(!empty($name)) {
            if(strpos($name,'.')) { // 支持 數據庫名.模型名的 定義
                list($this->dbName,$this->name) = explode('.',$name);
            }else{
                $this->name   =  $name;
            }
        }elseif(empty($this->name)){
            $this->name =   $this->getModelName();
        }
        // 設置表前綴
        if(is_null($tablePrefix)) {// 前綴為Null表示沒有前綴
            $this->tablePrefix = '';
        }elseif('' != $tablePrefix) {
            $this->tablePrefix = $tablePrefix;
        }elseif(!isset($this->tablePrefix)){
            $this->tablePrefix = C('DB_PREFIX');
        }

        // 數據庫初始化操作
        // 獲取數據庫操作對像
        // 當前模型有獨立的數據庫連接信息
        $this->db(0,empty($this->connection)?$connection:$this->connection,true);
    }

    /**
     * 自動檢測數據表信息
     * @access protected
     * @return void
     */
    protected function _checkTableInfo() {
        // 如果不是Model類 自動記錄數據表信息
        // 只在第一次執行記錄
        if(empty($this->fields)) {
            // 如果數據表字段沒有定義則自動獲取
            if(C('DB_FIELDS_CACHE')) {
                $db   =  $this->dbName?:C('DB_NAME');
                $fields = F('_fields/'.strtolower($db.'.'.$this->tablePrefix.$this->name));
                if($fields) {
                    $this->fields   =   $fields;
                    if(!empty($fields['_pk'])){
                        $this->pk       =   $fields['_pk'];
                    }
                    return ;
                }
            }
            // 每次都會讀取數據表信息
            $this->flush();
        }
    }

    /**
     * 獲取字段信息並緩存
     * @access public
     * @return void
     */
    public function flush() {
        // 緩存不存在則查詢數據表信息
        $this->db->setModel($this->name);
        $fields =   $this->db->getFields($this->getTableName());
        if(!$fields) { // 無法獲取字段信息
            return false;
        }
        $this->fields   =   array_keys($fields);
        unset($this->fields['_pk']);
        foreach ($fields as $key=>$val){
            // 記錄字段類型
            $type[$key]     =   $val['type'];
            if($val['primary']) {
                  // 增加復合主鍵支持
                if (isset($this->fields['_pk']) && $this->fields['_pk'] != null) {
                    if (is_string($this->fields['_pk'])) {
                        $this->pk   =   array($this->fields['_pk']);
                        $this->fields['_pk']   =   $this->pk;
                    }
                    $this->pk[]   =   $key;
                    $this->fields['_pk'][]   =   $key;
                } else {
                    $this->pk   =   $key;
                    $this->fields['_pk']   =   $key;
                }
                if($val['autoinc']) $this->autoinc   =   true;
            }
        }
        // 記錄字段類型信息
        $this->fields['_type'] =  $type;

        // 2008-3-7 增加緩存開關控制
        if(C('DB_FIELDS_CACHE')){
            // 永久緩存數據表信息
            $db   =  $this->dbName?:C('DB_NAME');
            F('_fields/'.strtolower($db.'.'.$this->tablePrefix.$this->name),$this->fields);
        }
    }

    /**
     * 設置數據對象的值
     * @access public
     * @param string $name 名稱
     * @param mixed $value 值
     * @return void
     */
    public function __set($name,$value) {
        // 設置數據對像屬性
        $this->data[$name]  =   $value;
    }

    /**
     * 獲取數據對象的值
     * @access public
     * @param string $name 名稱
     * @return mixed
     */
    public function __get($name) {
        return isset($this->data[$name])?$this->data[$name]:null;
    }

    /**
     * 檢測數據對象的值
     * @access public
     * @param string $name 名稱
     * @return boolean
     */
    public function __isset($name) {
        return isset($this->data[$name]);
    }

    /**
     * 銷毀數據對象的值
     * @access public
     * @param string $name 名稱
     * @return void
     */
    public function __unset($name) {
        unset($this->data[$name]);
    }

    /**
     * 利用__call方法實現一些特殊的Model方法
     * @access public
     * @param string $method 方法名稱
     * @param array $args 調用參數
     * @return mixed
     */
    public function __call($method,$args) {
        if(in_array(strtolower($method),$this->methods,true)) {
            // 連貫操作的實現
            $this->options[strtolower($method)] =   $args[0];
            return $this;
        }elseif(in_array(strtolower($method),array('count','sum','min','max','avg'),true)){
            // 統計查詢的實現
            $field =  isset($args[0])?$args[0]:'*';
            return $this->getField(strtoupper($method).'('.$field.') AS tp_'.$method);
        }elseif(strtolower(substr($method,0,5))=='getby') {
            // 根據某個字段獲取記錄
            $field   =   parse_name(substr($method,5));
            $where[$field] =  $args[0];
            return $this->where($where)->find();
        }elseif(strtolower(substr($method,0,10))=='getfieldby') {
            // 根據某個字段獲取記錄的某個值
            $name   =   parse_name(substr($method,10));
            $where[$name] =$args[0];
            return $this->where($where)->getField($args[1]);
        }elseif(isset($this->_scope[$method])){// 命名範圍的單獨調用支持
            return $this->scope($method,$args[0]);
        }else{
            E(__CLASS__.':'.$method.L('_METHOD_NOT_EXIST_'));
            return;
        }
    }
    // 回調方法 初始化模型
    protected function _initialize() {}

    /**
     * 對保存到數據庫的數據進行處理
     * @access protected
     * @param mixed $data 要操作的數據
     * @return boolean
     */
     protected function _facade($data) {

        // 檢查數據字段合法性
        if(!empty($this->fields)) {
            if(!empty($this->options['field'])) {
                $fields =   $this->options['field'];
                unset($this->options['field']);
                if(is_string($fields)) {
                    $fields =   explode(',',$fields);
                }    
            }else{
                $fields =   $this->fields;
            }        
            foreach ($data as $key=>$val){
                if(!in_array($key,$fields,true)){
                    if(!empty($this->options['strict'])){
                        E(L('_DATA_TYPE_INVALID_').':['.$key.'=>'.$val.']');
                    }
                    unset($data[$key]);
                }elseif(is_scalar($val)) {
                    // 字段類型檢查 和 強制轉換
                    $this->_parseType($data,$key);
                }
            }
        }
       
        // 安全過濾
        if(!empty($this->options['filter'])) {
            $data = array_map($this->options['filter'],$data);
            unset($this->options['filter']);
        }
        $this->_before_write($data);
        return $data;
     }

    // 寫入數據前的回調方法 包括新增和更新
    protected function _before_write(&$data) {}

    /**
     * 新增數據
     * @access public
     * @param mixed $data 數據
     * @param array $options 表達式
     * @param boolean $replace 是否replace
     * @return mixed
     */
    public function add($data='',$options=array(),$replace=false) {
        if(empty($data)) {
            // 沒有傳遞數據，獲取當前數據對象的值
            if(!empty($this->data)) {
                $data           =   $this->data;
                // 重置數據
                $this->data     = array();
            }else{
                $this->error    = L('_DATA_TYPE_INVALID_');
                return false;
            }
        }
        // 數據處理
        $data       =   $this->_facade($data);
        // 分析表達式
        $options    =   $this->_parseOptions($options);
        if(false === $this->_before_insert($data,$options)) {
            return false;
        }
        // 寫入數據到數據庫
        $result = $this->db->insert($data,$options,$replace);
        if(false !== $result && is_numeric($result)) {
            $pk     =   $this->getPk();
              // 增加復合主鍵支持
            if (is_array($pk)) return $result;
            $insertId   =   $this->getLastInsID();
            if($insertId) {
                // 自增主鍵返回插入ID
                $data[$pk]  = $insertId;
                if(false === $this->_after_insert($data,$options)) {
                    return false;
                }
                return $insertId;
            }
            if(false === $this->_after_insert($data,$options)) {
                return false;
            }
        }
        return $result;
    }
    // 插入數據前的回調方法
    protected function _before_insert(&$data,$options) {}
    // 插入成功後的回調方法
    protected function _after_insert($data,$options) {}

    public function addAll($dataList,$options=array(),$replace=false){
        if(empty($dataList)) {
            $this->error = L('_DATA_TYPE_INVALID_');
            return false;
        }
        // 數據處理
        foreach ($dataList as $key=>$data){
            $dataList[$key] = $this->_facade($data);
        }
        // 分析表達式
        $options =  $this->_parseOptions($options);
        // 寫入數據到數據庫
        $result = $this->db->insertAll($dataList,$options,$replace);
        if(false !== $result ) {
            $insertId   =   $this->getLastInsID();
            if($insertId) {
                return $insertId;
            }
        }
        return $result;
    }

    /**
     * 通過Select方式添加記錄
     * @access public
     * @param string $fields 要插入的數據表字段名
     * @param string $table 要插入的數據表名
     * @param array $options 表達式
     * @return boolean
     */
    public function selectAdd($fields='',$table='',$options=array()) {
        // 分析表達式
        $options =  $this->_parseOptions($options);
        // 寫入數據到數據庫
        if(false === $result = $this->db->selectInsert($fields?:$options['field'],$table?:$this->getTableName(),$options)){
            // 數據庫插入操作失敗
            $this->error = L('_OPERATION_WRONG_');
            return false;
        }else {
            // 插入成功
            return $result;
        }
    }

    /**
     * 保存數據
     * @access public
     * @param mixed $data 數據
     * @param array $options 表達式
     * @return boolean
     */
    public function save($data='',$options=array()) {
        if(empty($data)) {
            // 沒有傳遞數據，獲取當前數據對象的值
            if(!empty($this->data)) {
                $data           =   $this->data;
                // 重置數據
                $this->data     =   array();
            }else{
                $this->error    =   L('_DATA_TYPE_INVALID_');
                return false;
            }
        }
        // 數據處理
        $data       =   $this->_facade($data);
        if(empty($data)){
            // 沒有數據則不執行
            $this->error    =   L('_DATA_TYPE_INVALID_');
            return false;
        }
        // 分析表達式
        $options    =   $this->_parseOptions($options);
        $pk         =   $this->getPk();
        if(!isset($options['where']) ) {
            // 如果存在主鍵數據 則自動作為更新條件
            if (is_string($pk) && isset($data[$pk])) {
                $where[$pk]     =   $data[$pk];
                unset($data[$pk]);
            } elseif (is_array($pk)) {
                // 增加復合主鍵支持
                foreach ($pk as $field) {
                    if(isset($data[$field])) {
                        $where[$field]      =   $data[$field];
                    } else {
                           // 如果缺少復合主鍵數據則不執行
                        $this->error        =   L('_OPERATION_WRONG_');
                        return false;
                    }
                    unset($data[$field]);
                }
            }
            if(!isset($where)){
                // 如果沒有任何更新條件則不執行
                $this->error        =   L('_OPERATION_WRONG_');
                return false;
            }else{
                $options['where']   =   $where;
            }
        }

        if(is_array($options['where']) && isset($options['where'][$pk])){
            $pkValue    =   $options['where'][$pk];
        }
        if(false === $this->_before_update($data,$options)) {
            return false;
        }
        $result     =   $this->db->update($data,$options);
        if(false !== $result && is_numeric($result)) {
            if(isset($pkValue)) $data[$pk]   =  $pkValue;
            $this->_after_update($data,$options);
        }
        return $result;
    }
    // 更新數據前的回調方法
    protected function _before_update(&$data,$options) {}
    // 更新成功後的回調方法
    protected function _after_update($data,$options) {}

    /**
     * 刪除數據
     * @access public
     * @param mixed $options 表達式
     * @return mixed
     */
    public function delete($options=array()) {
        $pk   =  $this->getPk();
        if(empty($options) && empty($this->options['where'])) {
            // 如果刪除條件為空 則刪除當前數據對像所對應的記錄
            if(!empty($this->data) && isset($this->data[$pk]))
                return $this->delete($this->data[$pk]);
            else
                return false;
        }
        if(is_numeric($options)  || is_string($options)) {
            // 根據主鍵刪除記錄
            if(strpos($options,',')) {
                $where[$pk]     =  array('IN', $options);
            }else{
                $where[$pk]     =  $options;
            }
            $options            =  array();
            $options['where']   =  $where;
        }
        // 根據復合主鍵刪除記錄
        if (is_array($options) && (count($options) > 0) && is_array($pk)) {
            $count = 0;
            foreach (array_keys($options) as $key) {
                if (is_int($key)) $count++; 
            } 
            if ($count == count($pk)) {
                $i = 0;
                foreach ($pk as $field) {
                    $where[$field] = $options[$i];
                    unset($options[$i++]);
                }
                $options['where']  =  $where;
            } else {
                return false;
            }
        }
        // 分析表達式
        $options =  $this->_parseOptions($options);
        if(empty($options['where'])){
            // 如果條件為空 不進行刪除操作 除非設置 1=1
            return false;
        }        
        if(is_array($options['where']) && isset($options['where'][$pk])){
            $pkValue            =  $options['where'][$pk];
        }

        if(false === $this->_before_delete($options)) {
            return false;
        }        
        $result  =    $this->db->delete($options);
        if(false !== $result && is_numeric($result)) {
            $data = array();
            if(isset($pkValue)) $data[$pk]   =  $pkValue;
            $this->_after_delete($data,$options);
        }
        // 返回刪除記錄個數
        return $result;
    }
    // 刪除數據前的回調方法
    protected function _before_delete($options) {}    
    // 刪除成功後的回調方法
    protected function _after_delete($data,$options) {}

    /**
     * 查詢數據集
     * @access public
     * @param array $options 表達式參數
     * @return mixed
     */
    public function select($options=array()) {
        $pk   =  $this->getPk();
        if(is_string($options) || is_numeric($options)) {
            // 根據主鍵查詢
            if(strpos($options,',')) {
                $where[$pk]     =  array('IN',$options);
            }else{
                $where[$pk]     =  $options;
            }
            $options            =  array();
            $options['where']   =  $where;
        }elseif (is_array($options) && (count($options) > 0) && is_array($pk)) {
            // 根據復合主鍵查詢
            $count = 0;
            foreach (array_keys($options) as $key) {
                if (is_int($key)) $count++; 
            } 
            if ($count == count($pk)) {
                $i = 0;
                foreach ($pk as $field) {
                    $where[$field] = $options[$i];
                    unset($options[$i++]);
                }
                $options['where']  =  $where;
            } else {
                return false;
            }
        } elseif(false === $options){ // 用於子查詢 不查詢只返回SQL
            return  $this->buildSql();
        }
        // 分析表達式
        $options    =  $this->_parseOptions($options);
        // 判斷查詢緩存
        if(isset($options['cache'])){
            $cache  =   $options['cache'];
            $key    =   is_string($cache['key'])?$cache['key']:md5(serialize($options));
            $data   =   S($key,'',$cache);
            if(false !== $data){
                return $data;
            }
        }        
        $resultSet  = $this->db->select($options);
        if(false === $resultSet) {
            return false;
        }
        if(empty($resultSet)) { // 查詢結果為空
            return null;
        }

        if(is_string($resultSet)){
            return $resultSet;
        }

        $resultSet  =   array_map(array($this,'_read_data'),$resultSet);
        $this->_after_select($resultSet,$options);
        if(isset($options['index'])){ // 對數據集進行索引
            $index  =   explode(',',$options['index']);
            foreach ($resultSet as $result){
                $_key   =  $result[$index[0]];
                if(isset($index[1]) && isset($result[$index[1]])){
                    $cols[$_key] =  $result[$index[1]];
                }else{
                    $cols[$_key] =  $result;
                }
            }
            $resultSet  =   $cols;
        }
        if(isset($cache)){
            S($key,$resultSet,$cache);
        }
        return $resultSet;
    }
    // 查詢成功後的回調方法
    protected function _after_select(&$resultSet,$options) {}

    /**
     * 生成查詢SQL 可用於子查詢
     * @access public
     * @return string
     */
    public function buildSql() {
        return  '( '.$this->fetchSql(true)->select().' )';
    }

    /**
     * 分析表達式
     * @access protected
     * @param array $options 表達式參數
     * @return array
     */
    protected function _parseOptions($options=array()) {
        if(is_array($options))
            $options =  array_merge($this->options,$options);

        if(!isset($options['table'])){
            // 自動獲取表名
            $options['table']   =   $this->getTableName();
            $fields             =   $this->fields;
        }else{
            // 指定數據表 則重新獲取字段列表 但不支持類型檢測
            $fields             =   $this->getDbFields();
        }

        // 數據表別名
        if(!empty($options['alias'])) {
            $options['table']  .=   ' '.$options['alias'];
        }
        // 記錄操作的模型名稱
        $options['model']       =   $this->name;

        // 字段類型驗證
        if(isset($options['where']) && is_array($options['where']) && !empty($fields) && !isset($options['join'])) {
            // 對數組查詢條件進行字段類型檢查
            foreach ($options['where'] as $key=>$val){
                $key            =   trim($key);
                if(in_array($key,$fields,true)){
                    if(is_scalar($val)) {
                        $this->_parseType($options['where'],$key);
                    }
                }elseif(!is_numeric($key) && '_' != substr($key,0,1) && false === strpos($key,'.') && false === strpos($key,'(') && false === strpos($key,'|') && false === strpos($key,'&')){
                    if(!empty($this->options['strict'])){
                        E(L('_ERROR_QUERY_EXPRESS_').':['.$key.'=>'.$val.']');
                    } 
                    unset($options['where'][$key]);
                }
            }
        }
        // 查詢過後清空sql表達式組裝 避免影響下次查詢
        $this->options  =   array();
        // 表達式過濾
        $this->_options_filter($options);
        return $options;
    }
    // 表達式過濾回調方法
    protected function _options_filter(&$options) {}

    /**
     * 數據類型檢測
     * @access protected
     * @param mixed $data 數據
     * @param string $key 字段名
     * @return void
     */
    protected function _parseType(&$data,$key) {
        if(!isset($this->options['bind'][':'.$key]) && isset($this->fields['_type'][$key])){
            $fieldType = strtolower($this->fields['_type'][$key]);
            if(false !== strpos($fieldType,'enum')){
                // 支持ENUM類型優先檢測
            }elseif(false === strpos($fieldType,'bigint') && false !== strpos($fieldType,'int')) {
                $data[$key]   =  intval($data[$key]);
            }elseif(false !== strpos($fieldType,'float') || false !== strpos($fieldType,'double')){
                $data[$key]   =  floatval($data[$key]);
            }elseif(false !== strpos($fieldType,'bool')){
                $data[$key]   =  (bool)$data[$key];
            }
        }
    }

    /**
     * 數據讀取後的處理
     * @access protected
     * @param array $data 當前數據
     * @return array
     */
    protected function _read_data($data) {
        // 檢查字段映射
        if(!empty($this->_map) && C('READ_DATA_MAP')) {
            foreach ($this->_map as $key=>$val){
                if(isset($data[$val])) {
                    $data[$key] =   $data[$val];
                    unset($data[$val]);
                }
            }
        }
        return $data;
    }

    /**
     * 查詢數據
     * @access public
     * @param mixed $options 表達式參數
     * @return mixed
     */
    public function find($options=array()) {
        if(is_numeric($options) || is_string($options)) {
            $where[$this->getPk()]  =   $options;
            $options                =   array();
            $options['where']       =   $where;
        }		
        // 根據復合主鍵查找記錄
        $pk  =  $this->getPk();
        if (is_array($options) && (count($options) > 0) && is_array($pk)) {
            // 根據復合主鍵查詢
            $count = 0;
            foreach (array_keys($options) as $key) {
                if (is_int($key)) $count++; 
            } 
            if ($count == count($pk)) {
                $i = 0;
                foreach ($pk as $field) {
                    $where[$field] = $options[$i];
                    unset($options[$i++]);
                }
                $options['where']  =  $where;
            } else {
                return false;
            }
        }
        // 總是查找一條記錄
        $options['limit']   =   1;
        // 分析表達式
        $options            =   $this->_parseOptions($options);
        // 判斷查詢緩存
        if(isset($options['cache'])){
            $cache  =   $options['cache'];
            $key    =   is_string($cache['key'])?$cache['key']:md5(serialize($options));
            $data   =   S($key,'',$cache);
            if(false !== $data){
                $this->data     =   $data;
                return $data;
            }
        }
        $resultSet          =   $this->db->select($options);
        if(false === $resultSet) {
            return false;
        }
        if(empty($resultSet)) {// 查詢結果為空
            return null;
        }
        if(is_string($resultSet)){
            return $resultSet;
        }

        // 讀取數據後的處理
        $data   =   $this->_read_data($resultSet[0]);
        $this->_after_find($data,$options);
        if(!empty($this->options['result'])) {
            return $this->returnResult($data,$this->options['result']);
        }
        $this->data     =   $data;
        if(isset($cache)){
            S($key,$data,$cache);
        }
        return $this->data;
    }
    // 查詢成功的回調方法
    protected function _after_find(&$result,$options) {}

    protected function returnResult($data,$type=''){
        if ($type){
            if(is_callable($type)){
                return call_user_func($type,$data);
            }
            switch (strtolower($type)){
                case 'json':
                    return json_encode($data);
                case 'xml':
                    return xml_encode($data);
            }
        }
        return $data;
    }

    /**
     * 處理字段映射
     * @access public
     * @param array $data 當前數據
     * @param integer $type 類型 0 寫入 1 讀取
     * @return array
     */
    public function parseFieldsMap($data,$type=1) {
        // 檢查字段映射
        if(!empty($this->_map)) {
            foreach ($this->_map as $key=>$val){
                if($type==1) { // 讀取
                    if(isset($data[$val])) {
                        $data[$key] =   $data[$val];
                        unset($data[$val]);
                    }
                }else{
                    if(isset($data[$key])) {
                        $data[$val] =   $data[$key];
                        unset($data[$key]);
                    }
                }
            }
        }
        return $data;
    }

    /**
     * 設置記錄的某個字段值
     * 支持使用數據庫字段和方法
     * @access public
     * @param string|array $field  字段名
     * @param string $value  字段值
     * @return boolean
     */
    public function setField($field,$value='') {
        if(is_array($field)) {
            $data           =   $field;
        }else{
            $data[$field]   =   $value;
        }
        return $this->save($data);
    }

    /**
     * 字段值增長
     * @access public
     * @param string $field  字段名
     * @param integer $step  增長值
     * @param integer $lazyTime  延時時間(s)
     * @return boolean
     */
    public function setInc($field,$step=1,$lazyTime=0) {
        if($lazyTime>0) {// 延遲寫入
            $condition   =  $this->options['where'];
            $guid =  md5($this->name.'_'.$field.'_'.serialize($condition));
            $step = $this->lazyWrite($guid,$step,$lazyTime);
            if(false === $step ) return true; // 等待下次寫入
        }
        return $this->setField($field,array('exp',$field.'+'.$step));
    }

    /**
     * 字段值減少
     * @access public
     * @param string $field  字段名
     * @param integer $step  減少值
     * @param integer $lazyTime  延時時間(s)
     * @return boolean
     */
    public function setDec($field,$step=1,$lazyTime=0) {
        if($lazyTime>0) {// 延遲寫入
            $condition   =  $this->options['where'];
            $guid =  md5($this->name.'_'.$field.'_'.serialize($condition));
            $step = $this->lazyWrite($guid,$step,$lazyTime);
            if(false === $step ) return true; // 等待下次寫入
        }
        return $this->setField($field,array('exp',$field.'-'.$step));
    }

    /**
     * 延時更新檢查 返回false表示需要延時
     * 否則返回實際寫入的數值
     * @access public
     * @param string $guid  寫入標識
     * @param integer $step  寫入步進值
     * @param integer $lazyTime  延時時間(s)
     * @return false|integer
     */
    protected function lazyWrite($guid,$step,$lazyTime) {
        if(false !== ($value = S($guid))) { // 存在緩存寫入數據
            if(NOW_TIME > S($guid.'_time')+$lazyTime) {
                // 延時更新時間到了，刪除緩存數據 並實際寫入數據庫
                S($guid,NULL);
                S($guid.'_time',NULL);
                return $value+$step;
            }else{
                // 追加數據到緩存
                S($guid,$value+$step);
                return false;
            }
        }else{ // 沒有緩存數據
            S($guid,$step);
            // 計時開始
            S($guid.'_time',NOW_TIME);
            return false;
        }
    }

    /**
     * 獲取一條記錄的某個字段值
     * @access public
     * @param string $field  字段名
     * @param string $spea  字段數據間隔符號 NULL返回數組
     * @return mixed
     */
    public function getField($field,$sepa=null) {
        $options['field']       =   $field;
        $options                =   $this->_parseOptions($options);
        // 判斷查詢緩存
        if(isset($options['cache'])){
            $cache  =   $options['cache'];
            $key    =   is_string($cache['key'])?$cache['key']:md5($sepa.serialize($options));
            $data   =   S($key,'',$cache);
            if(false !== $data){
                return $data;
            }
        }        
        $field                  =   trim($field);
        if(strpos($field,',') && false !== $sepa) { // 多字段
            if(!isset($options['limit'])){
                $options['limit']   =   is_numeric($sepa)?$sepa:'';
            }
            $resultSet          =   $this->db->select($options);
            if(!empty($resultSet)) {
                $_field         =   explode(',', $field);
                $field          =   array_keys($resultSet[0]);
                $key1           =   array_shift($field);
                $key2           =   array_shift($field);
                $cols           =   array();
                $count          =   count($_field);
                foreach ($resultSet as $result){
                    $name   =  $result[$key1];
                    if(2==$count) {
                        $cols[$name]   =  $result[$key2];
                    }else{
                        $cols[$name]   =  is_string($sepa)?implode($sepa,array_slice($result,1)):$result;
                    }
                }
                if(isset($cache)){
                    S($key,$cols,$cache);
                }
                return $cols;
            }
        }else{   // 查找一條記錄
            // 返回數據個數
            if(true !== $sepa) {// 當sepa指定為true的時候 返回所有數據
                $options['limit']   =   is_numeric($sepa)?$sepa:1;
            }
            $result = $this->db->select($options);
            if(!empty($result)) {
                if(true !== $sepa && 1==$options['limit']) {
                    $data   =   reset($result[0]);
                    if(isset($cache)){
                        S($key,$data,$cache);
                    }            
                    return $data;
                }
                foreach ($result as $val){
                    $array[]    =   $val[$field];
                }
                if(isset($cache)){
                    S($key,$array,$cache);
                }                
                return $array;
            }
        }
        return null;
    }

    /**
     * 創建數據對像 但不保存到數據庫
     * @access public
     * @param mixed $data 創建數據
     * @param string $type 狀態
     * @return mixed
     */
     public function create($data='',$type='') {
        // 如果沒有傳值默認取POST數據
        if(empty($data)) {
            $data   =   I('post.');
        }elseif(is_object($data)){
            $data   =   get_object_vars($data);
        }
        // 驗證數據
        if(empty($data) || !is_array($data)) {
            $this->error = L('_DATA_TYPE_INVALID_');
            return false;
        }

        // 狀態
        $type = $type?:(!empty($data[$this->getPk()])?self::MODEL_UPDATE:self::MODEL_INSERT);

        // 檢查字段映射
        if(!empty($this->_map)) {
            foreach ($this->_map as $key=>$val){
                if(isset($data[$key])) {
                    $data[$val] =   $data[$key];
                    unset($data[$key]);
                }
            }
        }

        // 檢測提交字段的合法性
        if(isset($this->options['field'])) { // $this->field('field1,field2...')->create()
            $fields =   $this->options['field'];
            unset($this->options['field']);
        }elseif($type == self::MODEL_INSERT && isset($this->insertFields)) {
            $fields =   $this->insertFields;
        }elseif($type == self::MODEL_UPDATE && isset($this->updateFields)) {
            $fields =   $this->updateFields;
        }
        if(isset($fields)) {
            if(is_string($fields)) {
                $fields =   explode(',',$fields);
            }
            // 判斷令牌驗證字段
            if(C('TOKEN_ON'))   $fields[] = C('TOKEN_NAME', null, '__hash__');
            foreach ($data as $key=>$val){
                if(!in_array($key,$fields)) {
                    unset($data[$key]);
                }
            }
        }

        // 數據自動驗證
        if(!$this->autoValidation($data,$type)) return false;

        // 表單令牌驗證
        if(!$this->autoCheckToken($data)) {
            $this->error = L('_TOKEN_ERROR_');
            return false;
        }

        // 驗證完成生成數據對像
        if($this->autoCheckFields) { // 開啟字段檢測 則過濾非法字段數據
            $fields =   $this->getDbFields();
            foreach ($data as $key=>$val){
                if(!in_array($key,$fields)) {
                    unset($data[$key]);
                }elseif(MAGIC_QUOTES_GPC && is_string($val)){
                    $data[$key] =   stripslashes($val);
                }
            }
        }

        // 創建完成對數據進行自動處理
        $this->autoOperation($data,$type);
        // 賦值當前數據對像
        $this->data =   $data;
        // 返回創建的數據以供其他調用
        return $data;
     }

    // 自動表單令牌驗證
    // TODO  ajax無刷新多次提交暫不能滿足
    public function autoCheckToken($data) {
        // 支持使用token(false) 關閉令牌驗證
        if(isset($this->options['token']) && !$this->options['token']) return true;
        if(C('TOKEN_ON')){
            $name   = C('TOKEN_NAME', null, '__hash__');
            if(!isset($data[$name]) || !isset($_SESSION[$name])) { // 令牌數據無效
                return false;
            }

            // 令牌驗證
            list($key,$value)  =  explode('_',$data[$name]);
            if($value && $_SESSION[$name][$key] === $value) { // 防止重複提交
                unset($_SESSION[$name][$key]); // 驗證完成銷毀session
                return true;
            }
            // 開啟TOKEN重置
            if(C('TOKEN_RESET')) unset($_SESSION[$name][$key]);
            return false;
        }
        return true;
    }

    /**
     * 使用正則驗證數據
     * @access public
     * @param string $value  要驗證的數據
     * @param string $rule 驗證規則
     * @return boolean
     */
    public function regex($value,$rule) {
        $validate = array(
            'require'   =>  '/\S+/',
            'email'     =>  '/^\w+([-+.]\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$/',
            'url'       =>  '/^http(s?):\/\/(?:[A-za-z0-9-]+\.)+[A-za-z]{2,4}(:\d+)?(?:[\/\?#][\/=\?%\-&~`@[\]\':+!\.#\w]*)?$/',
            'currency'  =>  '/^\d+(\.\d+)?$/',
            'number'    =>  '/^\d+$/',
            'zip'       =>  '/^\d{6}$/',
            'integer'   =>  '/^[-\+]?\d+$/',
            'double'    =>  '/^[-\+]?\d+(\.\d+)?$/',
            'english'   =>  '/^[A-Za-z]+$/',
        );
        // 檢查是否有內置的正則表達式
        if(isset($validate[strtolower($rule)]))
            $rule       =   $validate[strtolower($rule)];
        return preg_match($rule,$value)===1;
    }

    /**
     * 自動表單處理
     * @access public
     * @param array $data 創建數據
     * @param string $type 創建類型
     * @return mixed
     */
    private function autoOperation(&$data,$type) {
        if(!empty($this->options['auto'])) {
            $_auto   =   $this->options['auto'];
            unset($this->options['auto']);
        }elseif(!empty($this->_auto)){
            $_auto   =   $this->_auto;
        }
        // 自動填充
        if(isset($_auto)) {
            foreach ($_auto as $auto){
                // 填充因子定義格式
                // array('field','填充內容','填充條件','附加規則',[額外參數])
                if(empty($auto[2])) $auto[2] =  self::MODEL_INSERT; // 默認為新增的時候自動填充
                if( $type == $auto[2] || $auto[2] == self::MODEL_BOTH) {
                    if(empty($auto[3])) $auto[3] =  'string';
                    switch(trim($auto[3])) {
                        case 'function':    //  使用函數進行填充 字段的值作為參數
                        case 'callback': // 使用回調方法
                            $args = isset($auto[4])?(array)$auto[4]:array();
                            if(isset($data[$auto[0]])) {
                                array_unshift($args,$data[$auto[0]]);
                            }
                            if('function'==$auto[3]) {
                                $data[$auto[0]]  = call_user_func_array($auto[1], $args);
                            }else{
                                $data[$auto[0]]  =  call_user_func_array(array(&$this,$auto[1]), $args);
                            }
                            break;
                        case 'field':    // 用其它字段的值進行填充
                            $data[$auto[0]] = $data[$auto[1]];
                            break;
                        case 'ignore': // 為空忽略
                            if($auto[1]===$data[$auto[0]])
                                unset($data[$auto[0]]);
                            break;
                        case 'string':
                        default: // 默認作為字符串填充
                            $data[$auto[0]] = $auto[1];
                    }
                    if(isset($data[$auto[0]]) && false === $data[$auto[0]] )   unset($data[$auto[0]]);
                }
            }
        }
        return $data;
    }

    /**
     * 自動表單驗證
     * @access protected
     * @param array $data 創建數據
     * @param string $type 創建類型
     * @return boolean
     */
    protected function autoValidation($data,$type) {
        if(!empty($this->options['validate'])) {
            $_validate   =   $this->options['validate'];
            unset($this->options['validate']);
        }elseif(!empty($this->_validate)){
            $_validate   =   $this->_validate;
        }
        // 屬性驗證
        if(isset($_validate)) { // 如果設置了數據自動驗證則進行數據驗證
            if($this->patchValidate) { // 重置驗證錯誤信息
                $this->error = array();
            }
            foreach($_validate as $key=>$val) {
                // 驗證因子定義格式
                // array(field,rule,message,condition,type,when,params)
                // 判斷是否需要執行驗證
                if(empty($val[5]) || ( $val[5]== self::MODEL_BOTH && $type < 3 ) || $val[5]== $type ) {
                    if(0==strpos($val[2],'{%') && strpos($val[2],'}'))
                        // 支持提示信息的多語言 使用 {%語言定義} 方式
                        $val[2]  =  L(substr($val[2],2,-1));
                    $val[3]  =  isset($val[3])?$val[3]:self::EXISTS_VALIDATE;
                    $val[4]  =  isset($val[4])?$val[4]:'regex';
                    // 判斷驗證條件
                    switch($val[3]) {
                        case self::MUST_VALIDATE:   // 必須驗證 不管表單是否有設置該字段
                            if(false === $this->_validationField($data,$val)) 
                                return false;
                            break;
                        case self::VALUE_VALIDATE:    // 值不為空的時候才驗證
                            if('' != trim($data[$val[0]]))
                                if(false === $this->_validationField($data,$val)) 
                                    return false;
                            break;
                        default:    // 默認表單存在該字段就驗證
                            if(isset($data[$val[0]]))
                                if(false === $this->_validationField($data,$val)) 
                                    return false;
                    }
                }
            }
            // 批量驗證的時候最後返回錯誤
            if(!empty($this->error)) return false;
        }
        return true;
    }

    /**
     * 驗證表單字段 支持批量驗證
     * 如果批量驗證返回錯誤的數組信息
     * @access protected
     * @param array $data 創建數據
     * @param array $val 驗證因子
     * @return boolean
     */
    protected function _validationField($data,$val) {
        if($this->patchValidate && isset($this->error[$val[0]]))
            return ; //當前字段已經有規則驗證沒有通過
        if(false === $this->_validationFieldItem($data,$val)){
            if($this->patchValidate) {
                $this->error[$val[0]]   =   $val[2];
            }else{
                $this->error            =   $val[2];
                return false;
            }
        }
        return ;
    }

    /**
     * 根據驗證因子驗證字段
     * @access protected
     * @param array $data 創建數據
     * @param array $val 驗證因子
     * @return boolean
     */
    protected function _validationFieldItem($data,$val) {
        switch(strtolower(trim($val[4]))) {
            case 'function':// 使用函數進行驗證
            case 'callback':// 調用方法進行驗證
                $args = isset($val[6])?(array)$val[6]:array();
                if(is_string($val[0]) && strpos($val[0], ','))
                    $val[0] = explode(',', $val[0]);
                if(is_array($val[0])){
                    // 支持多個字段驗證
                    foreach($val[0] as $field)
                        $_data[$field] = $data[$field];
                    array_unshift($args, $_data);
                }else{
                    array_unshift($args, $data[$val[0]]);
                }
                if('function'==$val[4]) {
                    return call_user_func_array($val[1], $args);
                }else{
                    return call_user_func_array(array(&$this, $val[1]), $args);
                }
            case 'confirm': // 驗證兩個字段是否相同
                return $data[$val[0]] == $data[$val[1]];
            case 'unique': // 驗證某個值是否唯一
                if(is_string($val[0]) && strpos($val[0],','))
                    $val[0]  =  explode(',',$val[0]);
                $map = array();
                if(is_array($val[0])) {
                    // 支持多個字段驗證
                    foreach ($val[0] as $field)
                        $map[$field]   =  $data[$field];
                }else{
                    $map[$val[0]] = $data[$val[0]];
                }
                $pk =   $this->getPk();
                if(!empty($data[$pk]) && is_string($pk)) { // 完善編輯的時候驗證唯一
                    $map[$pk] = array('neq',$data[$pk]);
                }
                if($this->where($map)->find())   return false;
                return true;
            default:  // 檢查附加規則
                return $this->check($data[$val[0]],$val[1],$val[4]);
        }
    }

    /**
     * 驗證數據 支持 in between equal length regex expire ip_allow ip_deny
     * @access public
     * @param string $value 驗證數據
     * @param mixed $rule 驗證表達式
     * @param string $type 驗證方式 默認為正則驗證
     * @return boolean
     */
    public function check($value,$rule,$type='regex'){
        $type   =   strtolower(trim($type));
        switch($type) {
            case 'in': // 驗證是否在某個指定範圍之內 逗號分隔字符串或者數組
            case 'notin':
                $range   = is_array($rule)? $rule : explode(',',$rule);
                return $type == 'in' ? in_array($value ,$range) : !in_array($value ,$range);
            case 'between': // 驗證是否在某個範圍
            case 'notbetween': // 驗證是否不在某個範圍            
                if (is_array($rule)){
                    $min    =    $rule[0];
                    $max    =    $rule[1];
                }else{
                    list($min,$max)   =  explode(',',$rule);
                }
                return $type == 'between' ? $value>=$min && $value<=$max : $value<$min || $value>$max;
            case 'equal': // 驗證是否等於某個值
            case 'notequal': // 驗證是否等於某個值            
                return $type == 'equal' ? $value == $rule : $value != $rule;
            case 'length': // 驗證長度
                $length  =  mb_strlen($value,'utf-8'); // 當前數據長度
                if(strpos($rule,',')) { // 長度區間
                    list($min,$max)   =  explode(',',$rule);
                    return $length >= $min && $length <= $max;
                }else{// 指定長度
                    return $length == $rule;
                }
            case 'expire':
                list($start,$end)   =  explode(',',$rule);
                if(!is_numeric($start)) $start   =  strtotime($start);
                if(!is_numeric($end)) $end   =  strtotime($end);
                return NOW_TIME >= $start && NOW_TIME <= $end;
            case 'ip_allow': // IP 操作許可驗證
                return in_array(get_client_ip(),explode(',',$rule));
            case 'ip_deny': // IP 操作禁止驗證
                return !in_array(get_client_ip(),explode(',',$rule));
            case 'regex':
            default:    // 默認使用正則驗證 可以使用驗證類中定義的驗證名稱
                // 檢查附加規則
                return $this->regex($value,$rule);
        }
    }

    /**
     * SQL查詢
     * @access public
     * @param string $sql  SQL指令
     * @param mixed $parse  是否需要解析SQL
     * @return mixed
     */
    public function query($sql,$parse=false) {
        if(!is_bool($parse) && !is_array($parse)) {
            $parse = func_get_args();
            array_shift($parse);
        }
        $sql  =   $this->parseSql($sql,$parse);
        return $this->db->query($sql);
    }

    /**
     * 執行SQL語句
     * @access public
     * @param string $sql  SQL指令
     * @param mixed $parse  是否需要解析SQL
     * @return false | integer
     */
    public function execute($sql,$parse=false) {
        if(!is_bool($parse) && !is_array($parse)) {
            $parse = func_get_args();
            array_shift($parse);
        }
        $sql  =   $this->parseSql($sql,$parse);
        return $this->db->execute($sql);
    }

    /**
     * 解析SQL語句
     * @access public
     * @param string $sql  SQL指令
     * @param boolean $parse  是否需要解析SQL
     * @return string
     */
    protected function parseSql($sql,$parse) {
        // 分析表達式
        if(true === $parse) {
            $options =  $this->_parseOptions();
            $sql    =   $this->db->parseSql($sql,$options);
        }elseif(is_array($parse)){ // SQL預處理
            $parse  =   array_map(array($this->db,'escapeString'),$parse);
            $sql    =   vsprintf($sql,$parse);
        }else{
            $sql    =   strtr($sql,array('__TABLE__'=>$this->getTableName(),'__PREFIX__'=>$this->tablePrefix));
            $prefix =   $this->tablePrefix;
            $sql    =   preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $sql);
        }
        $this->db->setModel($this->name);
        return $sql;
    }

    /**
     * 切換當前的數據庫連接
     * @access public
     * @param integer $linkNum  連接序號
     * @param mixed $config  數據庫連接信息
     * @param boolean $force 強制重新連接
     * @return Model
     */
    public function db($linkNum='',$config='',$force=false) {
        if('' === $linkNum && $this->db) {
            return $this->db;
        }

        if(!isset($this->_db[$linkNum]) || $force ) {
            // 創建一個新的實例
            if(!empty($config) && is_string($config) && false === strpos($config,'/')) { // 支持讀取配置參數
                $config  =  C($config);
            }
            $this->_db[$linkNum]            =    Db::getInstance($config);
        }elseif(NULL === $config){
            $this->_db[$linkNum]->close(); // 關閉數據庫連接
            unset($this->_db[$linkNum]);
            return ;
        }

        // 切換數據庫連接
        $this->db   =    $this->_db[$linkNum];
        $this->_after_db();
        // 字段檢測
        if(!empty($this->name) && $this->autoCheckFields)    $this->_checkTableInfo();
        return $this;
    }
    // 數據庫切換後回調方法
    protected function _after_db() {}

    /**
     * 得到當前的數據對像名稱
     * @access public
     * @return string
     */
    public function getModelName() {
        if(empty($this->name)){
            $name = substr(get_class($this),0,-strlen(C('DEFAULT_M_LAYER')));
            if ( $pos = strrpos($name,'\\') ) {//有命名空間
                $this->name = substr($name,$pos+1);
            }else{
                $this->name = $name;
            }
        }
        return $this->name;
    }

    /**
     * 得到完整的數據表名
     * @access public
     * @return string
     */
    public function getTableName() {
        if(empty($this->trueTableName)) {
            $tableName  = !empty($this->tablePrefix) ? $this->tablePrefix : '';
            if(!empty($this->tableName)) {
                $tableName .= $this->tableName;
            }else{
                $tableName .= parse_name($this->name);
            }
            $this->trueTableName    =   strtolower($tableName);
        }
        return (!empty($this->dbName)?$this->dbName.'.':'').$this->trueTableName;
    }

    /**
     * 啟動事務
     * @access public
     * @return void
     */
    public function startTrans() {
        $this->commit();
        $this->db->startTrans();
        return ;
    }

    /**
     * 提交事務
     * @access public
     * @return boolean
     */
    public function commit() {
        return $this->db->commit();
    }

    /**
     * 事務回滾
     * @access public
     * @return boolean
     */
    public function rollback() {
        return $this->db->rollback();
    }

    /**
     * 返回模型的錯誤信息
     * @access public
     * @return string
     */
    public function getError(){
        return $this->error;
    }

    /**
     * 返回數據庫的錯誤信息
     * @access public
     * @return string
     */
    public function getDbError() {
        return $this->db->getError();
    }

    /**
     * 返回最後插入的ID
     * @access public
     * @return string
     */
    public function getLastInsID() {
        return $this->db->getLastInsID();
    }

    /**
     * 返回最後執行的sql語句
     * @access public
     * @return string
     */
    public function getLastSql() {
        return $this->db->getLastSql($this->name);
    }
    // 鑒於getLastSql比較常用 增加_sql 別名
    public function _sql(){
        return $this->getLastSql();
    }

    /**
     * 獲取主鍵名稱
     * @access public
     * @return string
     */
    public function getPk() {
        return $this->pk;
    }

    /**
     * 獲取數據表字段信息
     * @access public
     * @return array
     */
    public function getDbFields(){
        if(isset($this->options['table'])) {// 動態指定表名
            if(is_array($this->options['table'])){
                $table  =   key($this->options['table']);
            }else{
                $table  =   $this->options['table'];
                if(strpos($table,')')){
                    // 子查詢
                    return false;
                }
            }
            $fields     =   $this->db->getFields($table);
            return  $fields ? array_keys($fields) : false;
        }
        if($this->fields) {
            $fields     =  $this->fields;
            unset($fields['_type'],$fields['_pk']);
            return $fields;
        }
        return false;
    }

    /**
     * 設置數據對像值
     * @access public
     * @param mixed $data 數據
     * @return Model
     */
    public function data($data=''){
        if('' === $data && !empty($this->data)) {
            return $this->data;
        }
        if(is_object($data)){
            $data   =   get_object_vars($data);
        }elseif(is_string($data)){
            parse_str($data,$data);
        }elseif(!is_array($data)){
            E(L('_DATA_TYPE_INVALID_'));
        }
        $this->data = $data;
        return $this;
    }

    /**
     * 指定當前的數據表
     * @access public
     * @param mixed $table
     * @return Model
     */
    public function table($table) {
        $prefix =   $this->tablePrefix;
        if(is_array($table)) {
            $this->options['table'] =   $table;
        }elseif(!empty($table)) {
            //將__TABLE_NAME__替換成帶前綴的表名
            $table  = preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $table);
            $this->options['table'] =   $table;
        }
        return $this;
    }

    /**
     * USING支持 用於多表刪除
     * @access public
     * @param mixed $using
     * @return Model
     */
    public function using($using){
        $prefix =   $this->tablePrefix;
        if(is_array($using)) {
            $this->options['using'] =   $using;
        }elseif(!empty($using)) {
            //將__TABLE_NAME__替換成帶前綴的表名
            $using  = preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $using);
            $this->options['using'] =   $using;
        }
        return $this;
    }

    /**
     * 查詢SQL組裝 join
     * @access public
     * @param mixed $join
     * @param string $type JOIN類型
     * @return Model
     */
    public function join($join,$type='INNER') {
        $prefix =   $this->tablePrefix;
        if(is_array($join)) {
            foreach ($join as $key=>&$_join){
                $_join  =   preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $_join);
                $_join  =   false !== stripos($_join,'JOIN')? $_join : $type.' JOIN ' .$_join;
            }
            $this->options['join']      =   $join;
        }elseif(!empty($join)) {
            //將__TABLE_NAME__字符串替換成帶前綴的表名
            $join  = preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $join);
            $this->options['join'][]    =   false !== stripos($join,'JOIN')? $join : $type.' JOIN '.$join;
        }
        return $this;
    }

    /**
     * 查詢SQL組裝 union
     * @access public
     * @param mixed $union
     * @param boolean $all
     * @return Model
     */
    public function union($union,$all=false) {
        if(empty($union)) return $this;
        if($all) {
            $this->options['union']['_all']  =   true;
        }
        if(is_object($union)) {
            $union   =  get_object_vars($union);
        }
        // 轉換union表達式
        if(is_string($union) ) {
            $prefix =   $this->tablePrefix;
            //將__TABLE_NAME__字符串替換成帶前綴的表名
            $options  = preg_replace_callback("/__([A-Z0-9_-]+)__/sU", function($match) use($prefix){ return $prefix.strtolower($match[1]);}, $union);
        }elseif(is_array($union)){
            if(isset($union[0])) {
                $this->options['union']  =  array_merge($this->options['union'],$union);
                return $this;
            }else{
                $options =  $union;
            }
        }else{
            E(L('_DATA_TYPE_INVALID_'));
        }
        $this->options['union'][]  =   $options;
        return $this;
    }

    /**
     * 查詢緩存
     * @access public
     * @param mixed $key
     * @param integer $expire
     * @param string $type
     * @return Model
     */
    public function cache($key=true,$expire=null,$type=''){
        // 增加快捷調用方式 cache(10) 等同於 cache(true, 10)
        if(is_numeric($key) && is_null($expire)){
            $expire = $key;
            $key    = true;
        }
        if(false !== $key)
            $this->options['cache']  =  array('key'=>$key,'expire'=>$expire,'type'=>$type);
        return $this;
    }

    /**
     * 指定查詢字段 支持字段排除
     * @access public
     * @param mixed $field
     * @param boolean $except 是否排除
     * @return Model
     */
    public function field($field,$except=false){
        if(true === $field) {// 獲取全部字段
            $fields     =  $this->getDbFields();
            $field      =  $fields?:'*';
        }elseif($except) {// 字段排除
            if(is_string($field)) {
                $field  =  explode(',',$field);
            }
            $fields     =  $this->getDbFields();
            $field      =  $fields?array_diff($fields,$field):$field;
        }
        $this->options['field']   =   $field;
        return $this;
    }

    /**
     * 調用命名範圍
     * @access public
     * @param mixed $scope 命名範圍名稱 支持多個 和直接定義
     * @param array $args 參數
     * @return Model
     */
    public function scope($scope='',$args=NULL){
        if('' === $scope) {
            if(isset($this->_scope['default'])) {
                // 默認的命名範圍
                $options    =   $this->_scope['default'];
            }else{
                return $this;
            }
        }elseif(is_string($scope)){ // 支持多個命名範圍調用 用逗號分割
            $scopes         =   explode(',',$scope);
            $options        =   array();
            foreach ($scopes as $name){
                if(!isset($this->_scope[$name])) continue;
                $options    =   array_merge($options,$this->_scope[$name]);
            }
            if(!empty($args) && is_array($args)) {
                $options    =   array_merge($options,$args);
            }
        }elseif(is_array($scope)){ // 直接傳入命名範圍定義
            $options        =   $scope;
        }
        
        if(is_array($options) && !empty($options)){
            $this->options  =   array_merge($this->options,array_change_key_case($options));
        }
        return $this;
    }

    /**
     * 指定查詢條件 支持安全過濾
     * @access public
     * @param mixed $where 條件表達式
     * @param mixed $parse 預處理參數
     * @return Model
     */
    public function where($where,$parse=null){
        if(!is_null($parse) && is_string($where)) {
            if(!is_array($parse)) {
                $parse = func_get_args();
                array_shift($parse);
            }
            $parse = array_map(array($this->db,'escapeString'),$parse);
            $where =   vsprintf($where,$parse);
        }elseif(is_object($where)){
            $where  =   get_object_vars($where);
        }
        if(is_string($where) && '' != $where){
            $map    =   array();
            $map['_string']   =   $where;
            $where  =   $map;
        }        
        if(isset($this->options['where'])){
            $this->options['where'] =   array_merge($this->options['where'],$where);
        }else{
            $this->options['where'] =   $where;
        }
        
        return $this;
    }

    /**
     * 指定查詢數量
     * @access public
     * @param mixed $offset 起始位置
     * @param mixed $length 查詢數量
     * @return Model
     */
    public function limit($offset,$length=null){
        if(is_null($length) && strpos($offset,',')){
            list($offset,$length)   =   explode(',',$offset);
        }
        $this->options['limit']     =   intval($offset).( $length? ','.intval($length) : '' );
        return $this;
    }

    /**
     * 指定分頁
     * @access public
     * @param mixed $page 頁數
     * @param mixed $listRows 每頁數量
     * @return Model
     */
    public function page($page,$listRows=null){
        if(is_null($listRows) && strpos($page,',')){
            list($page,$listRows)   =   explode(',',$page);
        }
        $this->options['page']      =   array(intval($page),intval($listRows));
        return $this;
    }

    /**
     * 查詢註釋
     * @access public
     * @param string $comment 註釋
     * @return Model
     */
    public function comment($comment){
        $this->options['comment'] =   $comment;
        return $this;
    }

    /**
     * 獲取執行的SQL語句
     * @access public
     * @param boolean $fetch 是否返回sql
     * @return Model
     */
    public function fetchSql($fetch){
        $this->options['fetch_sql'] =   $fetch;
        return $this;
    }

    /**
     * 參數綁定
     * @access public
     * @param string $key  參數名
     * @param mixed $value  綁定的變量及綁定參數
     * @return Model
     */
    public function bind($key,$value=false) {
        if(is_array($key)){
            $this->options['bind'] =    $key;
        }else{
            $num =  func_num_args();
            if($num>2){
                $params =   func_get_args();
                array_shift($params);
                $this->options['bind'][$key] =  $params;
            }else{
                $this->options['bind'][$key] =  $value;
            }        
        }
        return $this;
    }

    /**
     * 設置模型的屬性值
     * @access public
     * @param string $name 名稱
     * @param mixed $value 值
     * @return Model
     */
    public function setProperty($name,$value) {
        if(property_exists($this,$name))
            $this->$name = $value;
        return $this;
    }

}
