<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2006-2014 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------
namespace Think\Model;
use Think\Model;
/**
 * 高級模型擴展 
 */
class AdvModel extends Model {
    protected $optimLock        =   'lock_version';
    protected $returnType       =   'array';
    protected $blobFields       =   array();
    protected $blobValues       =   null;
    protected $serializeField   =   array();
    protected $readonlyField    =   array();
    protected $_filter          =   array();
    protected $partition        =   array();

    public function __construct($name='',$tablePrefix='',$connection='') {
        if('' !== $name || is_subclass_of($this,'AdvModel') ){
            // 如果是AdvModel子類或者有傳入模型名稱則獲取字段緩存
        }else{
            // 空的模型 關閉字段緩存
            $this->autoCheckFields = false;
        }
        parent::__construct($name,$tablePrefix,$connection);
    }

    /**
     * 利用__call方法重載 實現一些特殊的Model方法 （魔術方法）
     * @access public
     * @param string $method 方法名稱
     * @param mixed $args 調用參數
     * @return mixed
     */
    public function __call($method,$args) {
        if(strtolower(substr($method,0,3))=='top'){
            // 獲取前N條記錄
            $count = substr($method,3);
            array_unshift($args,$count);
            return call_user_func_array(array(&$this, 'topN'), $args);
        }else{
            return parent::__call($method,$args);
        }
    }

    /**
     * 對保存到數據庫的數據進行處理
     * @access protected
     * @param mixed $data 要操作的數據
     * @return boolean
     */
     protected function _facade($data) {
        // 檢查序列化字段
        $data = $this->serializeField($data);
        return parent::_facade($data);
     }

    // 查詢成功後的回調方法
    protected function _after_find(&$result,$options='') {
        // 檢查序列化字段
        $this->checkSerializeField($result);
        // 獲取文本字段
        $this->getBlobFields($result);
        // 檢查字段過濾
        $result   =  $this->getFilterFields($result);
        // 緩存樂觀鎖
        $this->cacheLockVersion($result);
    }

    // 查詢數據集成功後的回調方法
    protected function _after_select(&$resultSet,$options='') {
        // 檢查序列化字段
        $resultSet   =  $this->checkListSerializeField($resultSet);
        // 獲取文本字段
        $resultSet   =  $this->getListBlobFields($resultSet);
        // 檢查列表字段過濾
        $resultSet   =  $this->getFilterListFields($resultSet);
    }

    // 寫入前的回調方法
    protected function _before_insert(&$data,$options='') {
        // 記錄樂觀鎖
        $data = $this->recordLockVersion($data);
        // 檢查文本字段
        $data = $this->checkBlobFields($data);
        // 檢查字段過濾
        $data = $this->setFilterFields($data);
    }

    protected function _after_insert($data,$options) {
        // 保存文本字段
        $this->saveBlobFields($data);
    }

    // 更新前的回調方法
    protected function _before_update(&$data,$options='') {
        // 檢查樂觀鎖
        $pk     =   $this->getPK();
        if(isset($options['where'][$pk])){
            $id     =   $options['where'][$pk];   
            if(!$this->checkLockVersion($id,$data)) {
                return false;
            }
        }
        // 檢查文本字段
        $data = $this->checkBlobFields($data);
        // 檢查只讀字段
        $data = $this->checkReadonlyField($data);
        // 檢查字段過濾
        $data = $this->setFilterFields($data);
    }

    protected function _after_update($data,$options) {
        // 保存文本字段
        $this->saveBlobFields($data);
    }

    protected function _after_delete($data,$options) {
        // 刪除Blob數據
        $this->delBlobFields($data);
    }

    /**
     * 記錄樂觀鎖
     * @access protected
     * @param array $data 數據對像
     * @return array
     */
    protected function recordLockVersion($data) {
        // 記錄樂觀鎖
        if($this->optimLock && !isset($data[$this->optimLock]) ) {
            if(in_array($this->optimLock,$this->fields,true)) {
                $data[$this->optimLock]  =   0;
            }
        }
        return $data;
    }

    /**
     * 緩存樂觀鎖
     * @access protected
     * @param array $data 數據對像
     * @return void
     */
    protected function cacheLockVersion($data) {
        if($this->optimLock) {
            if(isset($data[$this->optimLock]) && isset($data[$this->getPk()])) {
                // 只有當存在樂觀鎖字段和主鍵有值的時候才記錄樂觀鎖
                $_SESSION[$this->name.'_'.$data[$this->getPk()].'_lock_version']    =   $data[$this->optimLock];
            }
        }
    }

    /**
     * 檢查樂觀鎖
     * @access protected
     * @param inteter $id  當前主鍵     
     * @param array $data  當前數據
     * @return mixed
     */
    protected function checkLockVersion($id,&$data) {
        // 檢查樂觀鎖
        $identify   = $this->name.'_'.$id.'_lock_version';
        if($this->optimLock && isset($_SESSION[$identify])) {
            $lock_version = $_SESSION[$identify];
            $vo   =  $this->field($this->optimLock)->find($id);
            $_SESSION[$identify]     =   $lock_version;
            $curr_version = $vo[$this->optimLock];
            if(isset($curr_version)) {
                if($curr_version>0 && $lock_version != $curr_version) {
                    // 記錄已經更新
                    $this->error = L('_RECORD_HAS_UPDATE_');
                    return false;
                }else{
                    // 更新樂觀鎖
                    $save_version = $data[$this->optimLock];
                    if($save_version != $lock_version+1) {
                        $data[$this->optimLock]  =   $lock_version+1;
                    }
                    $_SESSION[$identify]     =   $lock_version+1;
                }
            }
        }
        return true;
    }

    /**
     * 查找前N個記錄
     * @access public
     * @param integer $count 記錄個數
     * @param array $options 查詢表達式
     * @return array
     */
    public function topN($count,$options=array()) {
        $options['limit'] =  $count;
        return $this->select($options);
    }

    /**
     * 查詢符合條件的第N條記錄
     * 0 表示第一條記錄 -1 表示最後一條記錄
     * @access public
     * @param integer $position 記錄位置
     * @param array $options 查詢表達式
     * @return mixed
     */
    public function getN($position=0,$options=array()) {
        if($position>=0) { // 正向查找
            $options['limit'] = $position.',1';
            $list   =  $this->select($options);
            return $list?$list[0]:false;
        }else{ // 逆序查找
            $list   =  $this->select($options);
            return $list?$list[count($list)-abs($position)]:false;
        }
    }

    /**
     * 獲取滿足條件的第一條記錄
     * @access public
     * @param array $options 查詢表達式
     * @return mixed
     */
    public function first($options=array()) {
        return $this->getN(0,$options);
    }

    /**
     * 獲取滿足條件的最後一條記錄
     * @access public
     * @param array $options 查詢表達式
     * @return mixed
     */
    public function last($options=array()) {
        return $this->getN(-1,$options);
    }

    /**
     * 返回數據
     * @access public
     * @param array $data 數據
     * @param string $type 返回類型 默認為數組
     * @return mixed
     */
    public function returnResult($data,$type='') {
        if('' === $type)
            $type = $this->returnType;
        switch($type) {
            case 'array' :  return $data;
            case 'object':  return (object)$data;
            default:// 允許用戶自定義返回類型
                if(class_exists($type))
                    return new $type($data);
                else
                    E(L('_CLASS_NOT_EXIST_').':'.$type);
        }
    }

    /**
     * 獲取數據的時候過濾數據字段
     * @access protected
     * @param mixed $result 查詢的數據
     * @return array
     */
    protected function getFilterFields(&$result) {
        if(!empty($this->_filter)) {
            foreach ($this->_filter as $field=>$filter){
                if(isset($result[$field])) {
                    $fun  =  $filter[1];
                    if(!empty($fun)) {
                        if(isset($filter[2]) && $filter[2]){
                            // 傳遞整個數據對像作為參數
                            $result[$field]  =  call_user_func($fun,$result);
                        }else{
                            // 傳遞字段的值作為參數
                            $result[$field]  =  call_user_func($fun,$result[$field]);
                        }
                    }
                }
            }
        }
        return $result;
    }

    protected function getFilterListFields(&$resultSet) {
        if(!empty($this->_filter)) {
            foreach ($resultSet as $key=>$result)
                $resultSet[$key]  =  $this->getFilterFields($result);
        }
        return $resultSet;
    }

    /**
     * 寫入數據的時候過濾數據字段
     * @access protected
     * @param mixed $result 查詢的數據
     * @return array
     */
    protected function setFilterFields($data) {
        if(!empty($this->_filter)) {
            foreach ($this->_filter as $field=>$filter){
                if(isset($data[$field])) {
                    $fun              =  $filter[0];
                    if(!empty($fun)) {
                        if(isset($filter[2]) && $filter[2]) {
                            // 傳遞整個數據對像作為參數
                            $data[$field]   =  call_user_func($fun,$data);
                        }else{
                            // 傳遞字段的值作為參數
                            $data[$field]   =  call_user_func($fun,$data[$field]);
                        }
                    }
                }
            }
        }
        return $data;
    }

    /**
     * 返回數據列表
     * @access protected
     * @param array $resultSet 數據
     * @param string $type 返回類型 默認為數組
     * @return void
     */
    protected function returnResultSet(&$resultSet,$type='') {
        foreach ($resultSet as $key=>$data)
            $resultSet[$key]  =  $this->returnResult($data,$type);
        return $resultSet;
    }

    protected function checkBlobFields(&$data) {
        // 檢查Blob文件保存字段
        if(!empty($this->blobFields)) {
            foreach ($this->blobFields as $field){
                if(isset($data[$field])) {
                    if(isset($data[$this->getPk()]))
                        $this->blobValues[$this->name.'/'.$data[$this->getPk()].'_'.$field] =   $data[$field];
                    else
                        $this->blobValues[$this->name.'/@?id@_'.$field] =   $data[$field];
                    unset($data[$field]);
                }
            }
        }
        return $data;
    }

    /**
     * 獲取數據集的文本字段
     * @access protected
     * @param mixed $resultSet 查詢的數據
     * @param string $field 查詢的字段
     * @return void
     */
    protected function getListBlobFields(&$resultSet,$field='') {
        if(!empty($this->blobFields)) {
            foreach ($resultSet as $key=>$result){
                $result =   $this->getBlobFields($result,$field);
                $resultSet[$key]    =   $result;
            }
        }
        return $resultSet;
    }

    /**
     * 獲取數據的文本字段
     * @access protected
     * @param mixed $data 查詢的數據
     * @param string $field 查詢的字段
     * @return void
     */
    protected function getBlobFields(&$data,$field='') {
        if(!empty($this->blobFields)) {
            $pk =   $this->getPk();
            $id =   $data[$pk];
            if(empty($field)) {
                foreach ($this->blobFields as $field){
                    $identify   =   $this->name.'/'.$id.'_'.$field;
                    $data[$field]   =   F($identify);
                }
                return $data;
            }else{
                $identify   =   $this->name.'/'.$id.'_'.$field;
                return F($identify);
            }
        }
    }

    /**
     * 保存File方式的字段
     * @access protected
     * @param mixed $data 保存的數據
     * @return void
     */
    protected function saveBlobFields(&$data) {
        if(!empty($this->blobFields)) {
            foreach ($this->blobValues as $key=>$val){
                if(strpos($key,'@?id@'))
                    $key    =   str_replace('@?id@',$data[$this->getPk()],$key);
                F($key,$val);
            }
        }
    }

    /**
     * 刪除File方式的字段
     * @access protected
     * @param mixed $data 保存的數據
     * @param string $field 查詢的字段
     * @return void
     */
    protected function delBlobFields(&$data,$field='') {
        if(!empty($this->blobFields)) {
            $pk =   $this->getPk();
            $id =   $data[$pk];
            if(empty($field)) {
                foreach ($this->blobFields as $field){
                    $identify   =   $this->name.'/'.$id.'_'.$field;
                    F($identify,null);
                }
            }else{
                $identify   =   $this->name.'/'.$id.'_'.$field;
                F($identify,null);
            }
        }
    }

    /**
     * 檢查序列化數據字段
     * @access protected
     * @param array $data 數據
     * @return array
     */
     protected function serializeField(&$data) {
        // 檢查序列化字段
        if(!empty($this->serializeField)) {
            // 定義方式  $this->serializeField = array('ser'=>array('name','email'));
            foreach ($this->serializeField as $key=>$val){
                if(empty($data[$key])) {
                    $serialize  =   array();
                    foreach ($val as $name){
                        if(isset($data[$name])) {
                            $serialize[$name]   =   $data[$name];
                            unset($data[$name]);
                        }
                    }
                    if(!empty($serialize)) {
                        $data[$key] =   serialize($serialize);
                    }
                }
            }
        }
        return $data;
     }

    // 檢查返回數據的序列化字段
    protected function checkSerializeField(&$result) {
        // 檢查序列化字段
        if(!empty($this->serializeField)) {
            foreach ($this->serializeField as $key=>$val){
                if(isset($result[$key])) {
                    $serialize   =   unserialize($result[$key]);
                    foreach ($serialize as $name=>$value)
                        $result[$name]  =   $value;
                    unset($serialize,$result[$key]);
                }
            }
        }
        return $result;
    }

    // 檢查數據集的序列化字段
    protected function checkListSerializeField(&$resultSet) {
        // 檢查序列化字段
        if(!empty($this->serializeField)) {
            foreach ($this->serializeField as $key=>$val){
                foreach ($resultSet as $k=>$result){
                    if(isset($result[$key])) {
                        $serialize   =   unserialize($result[$key]);
                        foreach ($serialize as $name=>$value)
                            $result[$name]  =   $value;
                        unset($serialize,$result[$key]);
                        $resultSet[$k] =   $result;
                    }
                }
            }
        }
        return $resultSet;
    }

    /**
     * 檢查只讀字段
     * @access protected
     * @param array $data 數據
     * @return array
     */
    protected function checkReadonlyField(&$data) {
        if(!empty($this->readonlyField)) {
            foreach ($this->readonlyField as $key=>$field){
                if(isset($data[$field]))
                    unset($data[$field]);
            }
        }
        return $data;
    }

    /**
     * 批處理執行SQL語句
     * 批處理的指令都認為是execute操作
     * @access public
     * @param array $sql  SQL批處理指令
     * @return boolean
     */
    public function patchQuery($sql=array()) {
        if(!is_array($sql)) return false;
        // 自動啟動事務支持
        $this->startTrans();
        try{
            foreach ($sql as $_sql){
                $result   =  $this->execute($_sql);
                if(false === $result) {
                    // 發生錯誤自動回滾事務
                    $this->rollback();
                    return false;
                }
            }
            // 提交事務
            $this->commit();
        } catch (ThinkException $e) {
            $this->rollback();
        }
        return true;
    }

    /**
     * 得到分表的的數據表名
     * @access public
     * @param array $data 操作的數據
     * @return string
     */
    public function getPartitionTableName($data=array()) {
        // 對數據表進行分區
        if(isset($data[$this->partition['field']])) {
            $field   =   $data[$this->partition['field']];
            switch($this->partition['type']) {
                case 'id':
                    // 按照id範圍分表
                    $step    =   $this->partition['expr'];
                    $seq    =   floor($field / $step)+1;
                    break;
                case 'year':
                    // 按照年份分表
                    if(!is_numeric($field)) {
                        $field   =   strtotime($field);
                    }
                    $seq    =   date('Y',$field)-$this->partition['expr']+1;
                    break;
                case 'mod':
                    // 按照id的模數分表
                    $seq    =   ($field % $this->partition['num'])+1;
                    break;
                case 'md5':
                    // 按照md5的序列分表
                    $seq    =   (ord(substr(md5($field),0,1)) % $this->partition['num'])+1;
                    break;
                default :
                    if(function_exists($this->partition['type'])) {
                        // 支持指定函數哈希
                        $fun    =   $this->partition['type'];
                        $seq    =   (ord(substr($fun($field),0,1)) % $this->partition['num'])+1;
                    }else{
                        // 按照字段的首字母的值分表
                        $seq    =   (ord($field{0}) % $this->partition['num'])+1;
                    }
            }
            return $this->getTableName().'_'.$seq;
        }else{
            // 當設置的分表字段不在查詢條件或者數據中
            // 進行聯合查詢，必須設定 partition['num']
            $tableName  =   array();
            for($i=0;$i<$this->partition['num'];$i++)
                $tableName[] = 'SELECT * FROM '.$this->getTableName().'_'.($i+1);
            $tableName = '( '.implode(" UNION ",$tableName).') AS '.$this->name;
            return $tableName;
        }
    }
}