<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK ]
// +----------------------------------------------------------------------
// | Copyright (c) 2010 http://topthink.com All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------
namespace Think\Model;
use Think\Model;
/**
 * MongoModel模型類
 * 實現了ODM和ActiveRecords模式
 */
class MongoModel extends Model{
    // 主鍵類型
    const TYPE_OBJECT   = 1; 
    const TYPE_INT      = 2;
    const TYPE_STRING   = 3;

    // 主鍵名稱
    protected $pk               =   '_id';
    // _id 類型 1 Object 採用MongoId對像 2 Int 整形 支持自動增長 3 String 字符串Hash
    protected $_idType          =   self::TYPE_OBJECT;
    // 主鍵是否自增
    protected $_autoinc         =   true;
    // Mongo默認關閉字段檢測 可以動態追加字段
    protected $autoCheckFields  =   false;
    // 鏈操作方法列表
    protected $methods          =   array('table','order','auto','filter','validate');

    /**
     * 利用__call方法實現一些特殊的Model方法
     * @access public
     * @param string $method 方法名稱
     * @param array $args 調用參數
     * @return mixed
     */
    public function __call($method,$args) {
        if(in_array(strtolower($method),$this->methods,true)) {
            // 連貫操作的實現
            $this->options[strtolower($method)] =   $args[0];
            return $this;
        }elseif(strtolower(substr($method,0,5))=='getby') {
            // 根據某個字段獲取記錄
            $field   =   parse_name(substr($method,5));
            $where[$field] =$args[0];
            return $this->where($where)->find();
        }elseif(strtolower(substr($method,0,10))=='getfieldby') {
            // 根據某個字段獲取記錄的某個值
            $name   =   parse_name(substr($method,10));
            $where[$name] =$args[0];
            return $this->where($where)->getField($args[1]);
        }else{
            E(__CLASS__.':'.$method.L('_METHOD_NOT_EXIST_'));
            return;
        }
    }

    /**
     * 獲取字段信息並緩存 主鍵和自增信息直接配置
     * @access public
     * @return void
     */
    public function flush() {
        // 緩存不存在則查詢數據表信息
        $fields =   $this->db->getFields();
        if(!$fields) { // 暫時沒有數據無法獲取字段信息 下次查詢
            return false;
        }
        $this->fields   =   array_keys($fields);
        foreach ($fields as $key=>$val){
            // 記錄字段類型
            $type[$key]    =   $val['type'];
        }
        // 記錄字段類型信息
        if(C('DB_FIELDTYPE_CHECK'))   $this->fields['_type'] =  $type;

        // 2008-3-7 增加緩存開關控制
        if(C('DB_FIELDS_CACHE')){
            // 永久緩存數據表信息
            $db   =  $this->dbName?$this->dbName:C('DB_NAME');
            F('_fields/'.$db.'.'.$this->name,$this->fields);
        }
    }

    // 寫入數據前的回調方法 包括新增和更新
    protected function _before_write(&$data) {
        $pk   =  $this->getPk();
        // 根據主鍵類型處理主鍵數據
        if(isset($data[$pk]) && $this->_idType == self::TYPE_OBJECT) {
            $data[$pk] =  new \MongoId($data[$pk]);
        }    
    }

    /**
     * count統計 配合where連貫操作
     * @access public
     * @return integer
     */
    public function count(){
        // 分析表達式
        $options =  $this->_parseOptions();
        return $this->db->count($options);
    }

    /**
     * 獲取唯一值
     * @access public
     * @return array | false
     */
    public function distinct($field, $where=array() ){
        // 分析表達式
        $this->options =  $this->_parseOptions();
        $this->options['where'] = array_merge((array)$this->options['where'], $where);

        $command = array(
            "distinct" => $this->options['table'],
            "key" => $field,
            "query" => $this->options['where']
        );

        $result = $this->command($command);
        return isset($result['values']) ? $result['values'] : false;
    }

    /**
     * 獲取下一ID 用於自動增長型
     * @access public
     * @param string $pk 字段名 默認為主鍵
     * @return mixed
     */
    public function getMongoNextId($pk=''){
        if(empty($pk)) {
            $pk   =  $this->getPk();
        }
        return $this->db->getMongoNextId($pk);
    }

    /**
     * 新增數據
     * @access public
     * @param mixed $data 數據
     * @param array $options 表達式
     * @param boolean $replace 是否replace
     * @return mixed
     */
    public function add($data='',$options=array(),$replace=false) {
        if(empty($data)) {
            // 沒有傳遞數據，獲取當前數據對象的值
            if(!empty($this->data)) {
                $data           =   $this->data;
                // 重置數據
                $this->data     = array();
            }else{
                $this->error    = L('_DATA_TYPE_INVALID_');
                return false;
            }
        }
        // 分析表達式
        $options    =   $this->_parseOptions($options);
        // 數據處理
        $data       =   $this->_facade($data);
        if(false === $this->_before_insert($data,$options)) {
            return false;
        }
        // 寫入數據到數據庫
        $result = $this->db->insert($data,$options,$replace);
        if(false !== $result ) {
            $this->_after_insert($data,$options);
            if(isset($data[$this->getPk()])){
                return $data[$this->getPk()];
            }
        }
        return $result;
    }

    // 插入數據前的回調方法
    protected function _before_insert(&$data,$options) {
        // 寫入數據到數據庫
        if($this->_autoinc && $this->_idType== self::TYPE_INT) { // 主鍵自動增長
            $pk   =  $this->getPk();
            if(!isset($data[$pk])) {
                $data[$pk]   =  $this->db->getMongoNextId($pk);
            }
        }
    }

    public function clear(){
        return $this->db->clear();
    }

    // 查詢成功後的回調方法
    protected function _after_select(&$resultSet,$options) {
        array_walk($resultSet,array($this,'checkMongoId'));
    }

    /**
     * 獲取MongoId
     * @access protected
     * @param array $result 返回數據
     * @return array
     */
    protected function checkMongoId(&$result){
        if(is_object($result['_id'])) {
            $result['_id'] = $result['_id']->__toString();
        }
        return $result;
    }

    // 表達式過濾回調方法
    protected function _options_filter(&$options) {
        $id = $this->getPk();
        if(isset($options['where'][$id]) && is_scalar($options['where'][$id]) && $this->_idType== self::TYPE_OBJECT) {
            $options['where'][$id] = new \MongoId($options['where'][$id]);
        }
    }

    /**
     * 查詢數據
     * @access public
     * @param mixed $options 表達式參數
     * @return mixed
     */
     public function find($options=array()) {
         if( is_numeric($options) || is_string($options)) {
            $id   =  $this->getPk();
            $where[$id] = $options;
            $options = array();
            $options['where'] = $where;
         }
        // 分析表達式
        $options =  $this->_parseOptions($options);
        $result = $this->db->find($options);
        if(false === $result) {
            return false;
        }
        if(empty($result)) {// 查詢結果為空
            return null;
        }else{
            $this->checkMongoId($result);
        }
        $this->data = $result;
        $this->_after_find($this->data,$options);
        return $this->data;
     }

    /**
     * 字段值增長
     * @access public
     * @param string $field  字段名
     * @param integer $step  增長值
     * @return boolean
     */
    public function setInc($field,$step=1) {
        return $this->setField($field,array('inc',$step));
    }

    /**
     * 字段值減少
     * @access public
     * @param string $field  字段名
     * @param integer $step  減少值
     * @return boolean
     */
    public function setDec($field,$step=1) {
        return $this->setField($field,array('inc','-'.$step));
    }

    /**
     * 獲取一條記錄的某個字段值
     * @access public
     * @param string $field  字段名
     * @param string $spea  字段數據間隔符號
     * @return mixed
     */
    public function getField($field,$sepa=null) {
        $options['field']    =  $field;
        $options =  $this->_parseOptions($options);
        if(strpos($field,',')) { // 多字段
            if(is_numeric($sepa)) {// 限定數量
                $options['limit']   =   $sepa;
                $sepa   =   null;// 重置為null 返回數組
            }
            $resultSet = $this->db->select($options);
            if(!empty($resultSet)) {
                $_field = explode(',', $field);
                $field  = array_keys($resultSet[0]);
                $key =  array_shift($field);
                $key2 = array_shift($field);
                $cols   =   array();
                $count  =   count($_field);
                foreach ($resultSet as $result){
                    $name   =  $result[$key];
                    if(2==$count) {
                        $cols[$name]   =  $result[$key2];
                    }else{
                        $cols[$name]   =  is_null($sepa)?$result:implode($sepa,$result);
                    }
                }
                return $cols;
            }
        }else{
            // 返回數據個數
            if(true !== $sepa) {// 當sepa指定為true的時候 返回所有數據
                $options['limit']   =   is_numeric($sepa)?$sepa:1;
            }            // 查找一條記錄
            $result = $this->db->find($options);
            if(!empty($result)) {
                if(1==$options['limit']) return reset($result[0]);
                foreach ($result as $val){
                    $array[]    =   $val[$field];
                }
                return $array;
            }
        }
        return null;
    }

    /**
     * 執行Mongo指令
     * @access public
     * @param array $command  指令
     * @return mixed
     */
    public function command($command, $options=array()) {
        $options =  $this->_parseOptions($options);
        return $this->db->command($command, $options);
    }

    /**
     * 執行MongoCode
     * @access public
     * @param string $code  MongoCode
     * @param array $args   參數
     * @return mixed
     */
    public function mongoCode($code,$args=array()) {
        return $this->db->execute($code,$args);
    }

    // 數據庫切換後回調方法
    protected function _after_db() {
        // 切換Collection
        $this->db->switchCollection($this->getTableName(),$this->dbName?$this->dbName:C('db_name'));    
    }

    /**
     * 得到完整的數據表名 Mongo表名不帶dbName
     * @access public
     * @return string
     */
    public function getTableName() {
        if(empty($this->trueTableName)) {
            $tableName  = !empty($this->tablePrefix) ? $this->tablePrefix : '';
            if(!empty($this->tableName)) {
                $tableName .= $this->tableName;
            }else{
                $tableName .= parse_name($this->name);
            }
            $this->trueTableName    =   strtolower($tableName);
        }
        return $this->trueTableName;
    }

    /**
     * 分組查詢
     * @access public
     * @return string
     */
    public function group($key, $init, $reduce, $option=array()) {
        $option = $this->_parseOptions($option);

        //合併查詢條件
        if(isset($option['where']))
            $option['condition'] = array_merge((array)$option['condition'], $option['where']);

        return $this->db->group($key, $init, $reduce, $option);
    }

    /**
     * 返回Mongo運行錯誤信息
     * @access public
     * @return json
     */
    public function getLastError(){
        return $this->db->command(array('getLastError'=>1));
    }

    /**
     * 返回指定集合的統計信息，包括數據大小、已分配的存儲空間和索引的大小
     * @access public
     * @return json
     */
    public function status(){
        $option = $this->_parseOptions();
        return $this->db->command(array('collStats'=>$option['table']));
    }
    
    /**
     * 取得當前數據庫的對象
     * @access public
     * @return object
     */
    public function getDB(){
        return $this->db->getDB();
    }
    
    /**
     * 取得集合對象，可以進行創建索引等查詢
     * @access public
     * @return object
     */
    public function getCollection(){
        return $this->db->getCollection();
    }
}