<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2006-2014 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: 麥當苗兒 <zuojiazi@vip.qq.com> <http://www.zjzit.cn>
// +----------------------------------------------------------------------
namespace Think;
class Upload {
	/**
	 * 默認上傳配置
	 * @var array
	 */
	private $config = array('mimes' => array(), //允許上傳的文件MiMe類型
	'maxSize' => 0, //上傳的文件大小限制 (0-不做限制)
	'exts' => array(), //允許上傳的文件後綴
	'autoSub' => true, //自動子目錄保存文件
	'subName' => array('date', 'Y-m-d'), //子目錄創建方式，[0]-函數名，[1]-參數，多個參數使用數組
	'rootPath' => './Uploads/', //保存根路徑
	'savePath' => '', //保存路徑
	'saveName' => array('uniqid', ''), //上傳文件命名規則，[0]-函數名，[1]-參數，多個參數使用數組
	'saveExt' => '', //文件保存後綴，空則使用原後綴
	'replace' => false, //存在同名是否覆蓋
	'hash' => true, //是否生成hash編碼
	'callback' => false, //檢測文件是否存在回調，如果存在返回文件信息數組
	'driver' => '', // 文件上傳驅動
	'driverConfig' => array(), // 上傳驅動配置
	);

	/**
	 * 上傳錯誤信息
	 * @var string
	 */
	private $error = '';
	//上傳錯誤信息

	/**
	 * 上傳驅動實例
	 * @var Object
	 */
	private $uploader;

	/**
	 * 構造方法，用於構造上傳實例
	 * @param array  $config 配置
	 * @param string $driver 要使用的上傳驅動 LOCAL-本地上傳驅動，FTP-FTP上傳驅動
	 */
	public function __construct($config = array(), $driver = '', $driverConfig = null) {
		/* 獲取配置 */
		$this -> config = array_merge($this -> config, $config);

		/* 設置上傳驅動 */
		$this -> setDriver($driver, $driverConfig);

		/* 調整配置，把字符串配置參數轉換為數組 */
		if (!empty($this -> config['mimes'])) {
			if (is_string($this -> mimes)) {
				$this -> config['mimes'] = explode(',', $this -> mimes);
			}
			$this -> config['mimes'] = array_map('strtolower', $this -> mimes);
		}
		if (!empty($this -> config['exts'])) {
			if (is_string($this -> exts)) {
				$this -> config['exts'] = explode(',', $this -> exts);
			}
			$this -> config['exts'] = array_map('strtolower', $this -> exts);
		}
	}

	/**
	 * 使用 $this->name 獲取配置
	 * @param  string $name 配置名稱
	 * @return multitype    配置值
	 */
	public function __get($name) {
		return $this -> config[$name];
	}

	public function __set($name, $value) {
		if (isset($this -> config[$name])) {
			$this -> config[$name] = $value;
			if ($name == 'driverConfig') {
				//改變驅動配置後重置上傳驅動
				//注意：必須選改變驅動然後再改變驅動配置
				$this -> setDriver();
			}
		}
	}

	public function __isset($name) {
		return isset($this -> config[$name]);
	}

	/**
	 * 獲取最後一次上傳錯誤信息
	 * @return string 錯誤信息
	 */
	public function getError() {
		return $this -> error;
	}

	/**
	 * 上傳單個文件
	 * @param  array  $file 文件數組
	 * @return array        上傳成功後的文件信息
	 */
	public function uploadOne($file) {
		$info = $this -> upload(array($file));
		return $info ? $info[0] : $info;
	}

	/**
	 * 上傳文件
	 * @param 文件信息數組 $files ，通常是 $_FILES數組
	 */
	public function upload($files = '') {
		if ('' === $files) {
			$files = $_FILES;
		}
		if (empty($files)) {
			$this -> error = '沒有上傳的文件！';
			return false;
		}

		/* 檢測上傳根目錄 */
		if (!$this -> uploader -> checkRootPath($this -> rootPath)) {
			$this -> error = $this -> uploader -> getError();
			return false;
		}

		/* 檢查上傳目錄 */
		if (!$this -> uploader -> checkSavePath($this -> savePath)) {
			$this -> error = $this -> uploader -> getError();
			return false;
		}

		/* 逐個檢測並上傳文件 */
		$info = array();
		if (function_exists('finfo_open')) {
			$finfo = finfo_open(FILEINFO_MIME_TYPE);
		}
		// 對上傳文件數組信息處理
		$files = $this -> dealFiles($files);

		foreach ($files as $key => $file) {
			$file['name'] = strip_tags($file['name']);
			if (!isset($file['key']))
				$file['key'] = $key;
			/* 通過擴展獲取文件類型，可解決FLASH上傳$FILES數組返回文件類型錯誤的問題 */
			if (isset($finfo)) {
				$file['type'] = finfo_file($finfo, $file['tmp_name']);
			}

			/* 獲取上傳文件後綴，允許上傳無後綴文件 */
			$file['ext'] = pathinfo($file['name'], PATHINFO_EXTENSION);

			/* 文件上傳檢測 */
			if (!$this -> check($file)) {
				continue;
			}

			/* 獲取文件hash */
			if ($this -> hash) {
				$file['md5'] = md5_file($file['tmp_name']);
				$file['sha1'] = sha1_file($file['tmp_name']);
			}

			/* 調用回調函數檢測文件是否存在 */
			$data = call_user_func($this -> callback, $file);
			if ($this -> callback && $data) {
				if (file_exists('.' . $data['path'])) {
					$info[$key] = $data;
					continue;
				} elseif ($this -> removeTrash) {
					call_user_func($this -> removeTrash, $data);
					//刪除垃圾據
				}
			}

			/* 生成保存文件名 */
			$savename = $this -> getSaveName($file);
			if (false == $savename) {
				continue;
			} else {
				$file['savename'] = $savename;
			}

			/* 檢測並創建子目錄 */
			$subpath = $this -> getSubPath($file['name']);
			if (false === $subpath) {
				continue;
			} else {
				$file['savepath'] = $this -> savePath . $subpath;
			}

			/* 對圖像文件進行嚴格檢測 */
			$ext = strtolower($file['ext']);
			if (in_array($ext, array('gif', 'jpg', 'jpeg', 'bmp', 'png', 'swf'))) {
				$imginfo = getimagesize($file['tmp_name']);
				if (empty($imginfo) || ($ext == 'gif' && empty($imginfo['bits']))) {
					$this -> error = '非法圖像文件！';
					continue;
				}
			}

			/* 保存文件 並記錄保存成功的文件 */
			if ($this -> uploader -> save($file, $this -> replace)) {
				unset($file['error'], $file['tmp_name']);
				$info[$key] = $file;
			} else {
				$this -> error = $this -> uploader -> getError();
			}
		}
		if (isset($finfo)) {
			finfo_close($finfo);
		}
		return empty($info) ? false : $info;
	}

	/**
	 * 轉換上傳文件數組變量為正確的方式
	 * @access private
	 * @param array $files  上傳的文件變量
	 * @return array
	 */
	private function dealFiles($files) {
		$fileArray = array();
		$n = 0;
		foreach ($files as $key => $file) {
			if (is_array($file['name'])) {
				$keys = array_keys($file);
				$count = count($file['name']);
				for ($i = 0; $i < $count; $i++) {
					$fileArray[$n]['key'] = $key;
					foreach ($keys as $_key) {
						$fileArray[$n][$_key] = $file[$_key][$i];
					}
					$n++;
				}
			} else {
				$fileArray = $files;
				break;
			}
		}
		return $fileArray;
	}

	/**
	 * 設置上傳驅動
	 * @param string $driver 驅動名稱
	 * @param array $config 驅動配置
	 */
	private function setDriver($driver = null, $config = null) {
		$driver = $driver ? : ($this -> driver ? : C('FILE_UPLOAD_TYPE'));
		$config = $config ? : ($this -> driverConfig ? : C('UPLOAD_TYPE_CONFIG'));
		$class = strpos($driver, '\\') ? $driver : 'Think\\Upload\\Driver\\' . ucfirst(strtolower($driver));
		$this -> uploader = new $class($config);
		if (!$this -> uploader) {
			E("不存在上傳驅動：{$name}");
		}
	}

	/**
	 * 檢查上傳的文件
	 * @param array $file 文件信息
	 */
	private function check($file) {
		/* 文件上傳失敗，捕獲錯誤代碼 */

		if ($file['error']) {
			$this -> error($file['error']);
			return false;
		}

		/* 無效上傳 */
		if (empty($file['name'])) {
			$this -> error = '未知上傳錯誤！';
		}

		if (!$file['is_move']) {
			/* 檢查是否合法上傳 */
			if (!is_uploaded_file($file['tmp_name'])) {
				$this -> error = '非法上傳文件！';
				return false;
			}
		}

		/* 檢查文件大小 */
		if (!$this -> checkSize($file['size'])) {
			$this -> error = '上傳文件大小不符！';
			return false;
		}

		/* 檢查文件Mime類型 */
		//TODO:FLASH上傳的文件獲取到的mime類型都為application/octet-stream
		if (!$this -> checkMime($file['type'])) {
			$this -> error = '上傳文件MIME類型不允許！';
			return false;
		}

		/* 檢查文件後綴 */
		if (!$this -> checkExt($file['ext'])) {
			$this -> error = '上傳文件後綴不允許';
			return false;
		}

		/* 通過檢測 */
		return true;
	}

	/**
	 * 獲取錯誤代碼信息
	 * @param string $errorNo  錯誤號
	 */
	private function error($errorNo) {
		switch ($errorNo) {
			case 1 :
				$this -> error = '上傳的文件超過了 php.ini 中 upload_max_filesize 選項限制的值！';
				break;
			case 2 :
				$this -> error = '上傳文件的大小超過了 HTML 表單中 MAX_FILE_SIZE 選項指定的值！';
				break;
			case 3 :
				$this -> error = '文件只有部分被上傳！';
				break;
			case 4 :
				$this -> error = '沒有文件被上傳！';
				break;
			case 6 :
				$this -> error = '找不到臨時文件夾！';
				break;
			case 7 :
				$this -> error = '文件寫入失敗！';
				break;
			default :
				$this -> error = '未知上傳錯誤！';
		}
	}

	/**
	 * 檢查文件大小是否合法
	 * @param integer $size 數據
	 */
	private function checkSize($size) {
		return !($size > $this -> maxSize) || (0 == $this -> maxSize);
	}

	/**
	 * 檢查上傳的文件MIME類型是否合法
	 * @param string $mime 數據
	 */
	private function checkMime($mime) {
		return empty($this -> config['mimes']) ? true : in_array(strtolower($mime), $this -> mimes);
	}

	/**
	 * 檢查上傳的文件後綴是否合法
	 * @param string $ext 後綴
	 */
	private function checkExt($ext) {
		return empty($this -> config['exts']) ? true : in_array(strtolower($ext), $this -> exts);
	}

	/**
	 * 根據上傳文件命名規則取得保存文件名
	 * @param string $file 文件信息
	 */
	private function getSaveName($file) {
		$rule = $this -> saveName;
		if (empty($rule)) {//保持文件名不變
			/* 解決pathinfo中文文件名BUG */
			$filename = substr(pathinfo("_{$file['name']}", PATHINFO_FILENAME), 1);
			$savename = $filename;
		} else {
			$savename = $this -> getName($rule, $file['name']);
			if (empty($savename)) {
				$this -> error = '文件命名規則錯誤！';
				return false;
			}
		}

		/* 文件保存後綴，支持強制更改文件後綴 */
		$ext = empty($this -> config['saveExt']) ? $file['ext'] : $this -> saveExt;

		return $savename . '.' . $ext;
	}

	/**
	 * 獲取子目錄的名稱
	 * @param array $file  上傳的文件信息
	 */
	private function getSubPath($filename) {
		$subpath = '';
		$rule = $this -> subName;
		if ($this -> autoSub && !empty($rule)) {
			$subpath = $this -> getName($rule, $filename) . '/';

			if (!empty($subpath) && !$this -> uploader -> mkdir($this -> savePath . $subpath)) {
				$this -> error = $this -> uploader -> getError();
				return false;
			}
		}
		return $subpath;
	}

	/**
	 * 根據指定的規則獲取文件或目錄名稱
	 * @param  array  $rule     規則
	 * @param  string $filename 原文件名
	 * @return string           文件或目錄名稱
	 */
	private function getName($rule, $filename) {
		$name = '';
		if (is_array($rule)) {//數組規則
			$func = $rule[0];
			$param = (array)$rule[1];
			foreach ($param as &$value) {
				$value = str_replace('__FILE__', $filename, $value);
			}
			$name = call_user_func_array($func, $param);
		} elseif (is_string($rule)) {//字符串規則
			if (function_exists($rule)) {
				$name = call_user_func($rule);
			} else {
				$name = $rule;
			}
		}
		return $name;
	}

}
