<?php 
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace GatewayWorker;

use Workerman\Connection\TcpConnection;

use \Workerman\Worker;
use \Workerman\Lib\Timer;
use \Workerman\Protocols\GatewayProtocol;
use \GatewayWorker\Lib\Lock;
use \GatewayWorker\Lib\Store;
use \Workerman\Autoloader;

/**
 * 
 * Gateway，基於Worker開發
 * 用於轉發客戶端的數據給Worker處理，以及轉發Worker的數據給客戶端
 * 
 * @author walkor<walkor@workerman.net>
 *
 */
class Gateway extends Worker
{
    /**
     * 本機ip
     * @var 單機部署默認127.0.0.1，如果是分佈式部署，需要設置成本機ip
     */
    public $lanIp = '127.0.0.1';
    
    /**
     * gateway內部通訊起始端口，每個gateway實例應該都不同，步長1000
     * @var int
     */
    public $startPort = 2000;
    
    /**
     * 是否可以平滑重啟，gateway不能平滑重啟，否則會導致連接斷開
     * @var bool
     */
    public $reloadable = false;
    
    /**
     * 心跳時間間隔
     * @var int
     */
    public $pingInterval = 0;

    /**
     * $pingNotResponseLimit*$pingInterval時間內，客戶端未發送任何數據，斷開客戶端連接
     * @var int
     */
    public $pingNotResponseLimit = 0;
    
    /**
     * 服務端向客戶端發送的心跳數據
     * @var string
     */
    public $pingData = '';
    
    /**
     * 路由函數
     * @var callback
     */
    public $router = null;
    
    /**
     * 保存客戶端的所有connection對像
     * @var array
     */
    protected $_clientConnections = array();
    
    /**
     * 保存所有worker的內部連接的connection對像
     * @var array
     */
    protected $_workerConnections = array();
    
    /**
     * gateway內部監聽worker內部連接的worker
     * @var Worker
     */
    protected $_innerTcpWorker = null;
    
    /**
     * gateway內部監聽udp數據的worker
     * @var Worker
     */
    protected $_innerUdpWorker = null;
    
    /**
     * 當worker啟動時
     * @var callback
     */
    protected $_onWorkerStart = null;
    
    /**
     * 當有客戶端連接時
     * @var callback
     */
    protected $_onConnect = null;
    
    /**
     * 當客戶端發來消息時
     * @var callback
     */
    protected $_onMessage = null;
    
    /**
     * 當客戶端連接關閉時
     * @var callback
     */
    protected $_onClose = null;
    
    /**
     * 當worker停止時
     * @var callback
     */
    protected $_onWorkerStop = null;
    
    /**
     * 進程啟動時間
     * @var int
     */
    protected $_startTime = 0;
    
    /**
     * 構造函數
     * @param string $socket_name
     * @param array $context_option
     */
    public function __construct($socket_name, $context_option = array())
    {
        parent::__construct($socket_name, $context_option);
        
        $this->router = array("\\GatewayWorker\\Gateway", 'routerRand');
        
        $backrace = debug_backtrace();
        $this->_appInitPath = dirname($backrace[0]['file']);
    }
    
    /**
     * 運行
     * @see Workerman.Worker::run()
     */
    public function run()
    {
        // 保存用戶的回調，當對應的事件發生時觸發
        $this->_onWorkerStart = $this->onWorkerStart;
        $this->onWorkerStart = array($this, 'onWorkerStart');
        // 保存用戶的回調，當對應的事件發生時觸發
        $this->_onConnect = $this->onConnect;
        $this->onConnect = array($this, 'onClientConnect');
        
        // onMessage禁止用戶設置回調
        $this->onMessage = array($this, 'onClientMessage');
        
        // 保存用戶的回調，當對應的事件發生時觸發
        $this->_onClose = $this->onClose;
        $this->onClose = array($this, 'onClientClose');
        // 保存用戶的回調，當對應的事件發生時觸發
        $this->_onWorkerStop = $this->onWorkerStop;
        $this->onWorkerStop = array($this, 'onWorkerStop');
        
        // 記錄進程啟動的時間
        $this->_startTime = time();
        // 運行父方法
        parent::run();
    }
    
    /**
     * 當客戶端發來數據時，轉發給worker處理
     * @param TcpConnection $connection
     * @param mixed $data
     */
    public function onClientMessage($connection, $data)
    {
        $connection->pingNotResponseCount = -1;
        $this->sendToWorker(GatewayProtocol::CMD_ON_MESSAGE, $connection, $data);
    }
    
    /**
     * 當客戶端連接上來時，初始化一些客戶端的數據
     * 包括全局唯一的client_id、初始化session等
     * @param unknown_type $connection
     */
    public function onClientConnect($connection)
    {
        // 分配一個全局唯一的client_id
        $connection->globalClientId = $this->createGlobalClientId();
        // 保存該連接的內部通訊的數據包報頭，避免每次重新初始化
        $connection->gatewayHeader = array(
            'local_ip' => $this->lanIp,
            'local_port' => $this->lanPort,
            'client_ip'=>$connection->getRemoteIp(),
            'client_port'=>$connection->getRemotePort(),
            'client_id'=>$connection->globalClientId,
        );
        // 連接的session
        $connection->session = '';
        // 該連接的心跳參數
        $connection->pingNotResponseCount = -1;
        // 保存客戶端連接connection對像
        $this->_clientConnections[$connection->globalClientId] = $connection;
        // 保存該連接的內部gateway通訊地址
        $address = $this->lanIp.':'.$this->lanPort;
        $this->storeClientAddress($connection->globalClientId, $address);
        
        // 如果用戶有自定義onConnect回調，則執行
        if($this->_onConnect)
        {
            call_user_func($this->_onConnect, $connection);
        }
        
        // 如果設置了Event::onConnect，則通知worker進程，讓worker執行onConnect
        if(method_exists('Event','onConnect'))
        {
            $this->sendToWorker(GatewayProtocol::CMD_ON_CONNECTION, $connection);
        }
    }
    
    /**
     * 發送數據給worker進程
     * @param int $cmd
     * @param TcpConnection $connection
     * @param mixed $body
     */
    protected function sendToWorker($cmd, $connection, $body = '')
    {
        $gateway_data = $connection->gatewayHeader;
        $gateway_data['cmd'] = $cmd;
        $gateway_data['body'] = $body;
        $gateway_data['ext_data'] = $connection->session;
        if($this->_workerConnections)
        {
            // 調用路由函數，選擇一個worker把請求轉發給它
            $worker_connection = call_user_func($this->router, $this->_workerConnections, $connection, $cmd, $body);
            if(false === $worker_connection->send($gateway_data))
            {
                $msg = "SendBufferToWorker fail. May be the send buffer are overflow";
                $this->log($msg);
                return false;
            }
        }
        // 沒有可用的worker
        else
        {
            // gateway啟動後1-2秒內SendBufferToWorker fail是正常現象，因為與worker的連接還沒建立起來，所以不記錄日誌，只是關閉連接
            $time_diff = 2;
            if(time() - $this->_startTime >= $time_diff)
            {
                $msg = "SendBufferToWorker fail. The connections between Gateway and BusinessWorker are not ready";
                $this->log($msg);
            }
            $connection->destroy();
            return false;
        }
        return true;
    }
    
    /**
     * 隨機路由，返回worker connection對像
     * @param array $worker_connections
     * @param TcpConnection $client_connection
     * @param int $cmd
     * @param mixed $buffer
     * @return TcpConnection
     */
    public static function routerRand($worker_connections, $client_connection, $cmd, $buffer)
    {
        return $worker_connections[array_rand($worker_connections)];
    }
    
    /**
     * 保存客戶端連接的gateway通訊地址
     * @param int $global_client_id
     * @param string $address
     * @return bool
     */
    protected function storeClientAddress($global_client_id, $address)
    {
        if(!Store::instance('gateway')->set('client_id-'.$global_client_id, $address))
        {
            $msg = 'storeClientAddress fail.';
            if(get_class(Store::instance('gateway')) == 'Memcached')
            {
                $msg .= " reason :".Store::instance('gateway')->getResultMessage();
            }
            $this->log($msg);
            return false;
        }
        return true;
    }
    
    /**
     * 刪除客戶端gateway通訊地址
     * @param int $global_client_id
     * @return void
     */
    protected function delClientAddress($global_client_id)
    {
        Store::instance('gateway')->delete('client_id-'.$global_client_id);
    }
    
    /**
     * 當客戶端關閉時
     * @param unknown_type $connection
     */
    public function onClientClose($connection)
    {
        // 嘗試通知worker，觸發Event::onClose
        if(method_exists('Event','onClose'))
        {
            $this->sendToWorker(GatewayProtocol::CMD_ON_CLOSE, $connection);
        }
        // 清理連接的數據
        $this->delClientAddress($connection->globalClientId);
        unset($this->_clientConnections[$connection->globalClientId]);
        if($this->_onClose)
        {
            call_user_func($this->_onClose, $connection);
        }
    }
    
    /**
     * 創建一個workerman集群全局唯一的client_id
     * @return int|false
     */
    protected function createGlobalClientId()
    {
        $global_socket_key = 'GLOBAL_CLIENT_ID_KEY';
        $store = Store::instance('gateway');
        $global_client_id = $store->increment($global_socket_key);
        if(!$global_client_id || $global_client_id > 2147483646)
        {
            $store->set($global_socket_key, 0);
            $global_client_id = $store->increment($global_socket_key);
        }
    
        if(!$global_client_id)
        {
            $msg = "createGlobalClientId fail :";
            if(get_class($store) == 'Memcached')
            {
                $msg .= $store->getResultMessage();
            }
            $this->log($msg);
        }
        
        return $global_client_id;
    }
    
    /**
     * 當Gateway啟動的時候觸發的回調函數
     * @return void
     */
    public function onWorkerStart()
    {
        // 分配一個內部通訊端口
        $this->lanPort = function_exists('posix_getppid') ? $this->startPort - posix_getppid() + posix_getpid() : $this->startPort;
        if($this->lanPort<0 || $this->lanPort >=65535)
        {
            $this->lanPort = rand($this->startPort, 65535);
        }
        
        // 如果有設置心跳，則定時執行
        if($this->pingInterval > 0)
        {
            Timer::add($this->pingInterval, array($this, 'ping'));
        }
    
        // 初始化gateway內部的監聽，用於監聽worker的連接已經連接上發來的數據
        $this->_innerTcpWorker = new Worker("GatewayProtocol://{$this->lanIp}:{$this->lanPort}");
        $this->_innerTcpWorker->listen();
        $this->_innerUdpWorker = new Worker("GatewayProtocol://{$this->lanIp}:{$this->lanPort}");
        $this->_innerUdpWorker->transport = 'udp';
        $this->_innerUdpWorker->listen();
    
        // 重新設置自動加載根目錄
        Autoloader::setRootPath($this->_appInitPath);
        
        // 設置內部監聽的相關回調
        $this->_innerTcpWorker->onMessage = array($this, 'onWorkerMessage');
        $this->_innerUdpWorker->onMessage = array($this, 'onWorkerMessage');
        
        $this->_innerTcpWorker->onConnect = array($this, 'onWorkerConnect');
        $this->_innerTcpWorker->onClose = array($this, 'onWorkerClose');
        
        // 註冊gateway的內部通訊地址，worker去連這個地址，以便gateway與worker之間建立起TCP長連接
        if(!$this->registerAddress())
        {
            $this->log('registerAddress fail and exit');
            Worker::stopAll();
        }
        
        if($this->_onWorkerStart)
        {
            call_user_func($this->_onWorkerStart, $this);
        }
    }
    
    
    /**
     * 當worker通過內部通訊端口連接到gateway時
     * @param TcpConnection $connection
     */
    public function onWorkerConnect($connection)
    {
        $connection->remoteAddress = $connection->getRemoteIp().':'.$connection->getRemotePort();
        if(TcpConnection::$defaultMaxSendBufferSize == $connection->maxSendBufferSize)
        {
            $connection->maxSendBufferSize = 10*1024*1024;
        }
        $this->_workerConnections[$connection->remoteAddress] = $connection;
    }
    
    /**
     * 當worker發來數據時
     * @param TcpConnection $connection
     * @param mixed $data
     * @throws \Exception
     */
    public function onWorkerMessage($connection, $data)
    {
        $cmd = $data['cmd'];
        switch($cmd)
        {
            // 向某客戶端發送數據，Gateway::sendToClient($client_id, $message);
            case GatewayProtocol::CMD_SEND_TO_ONE:
                if(isset($this->_clientConnections[$data['client_id']]))
                {
                    $this->_clientConnections[$data['client_id']]->send($data['body']);
                }
                break;
                // 關閉客戶端連接，Gateway::closeClient($client_id);
            case GatewayProtocol::CMD_KICK:
                if(isset($this->_clientConnections[$data['client_id']]))
                {
                    $this->_clientConnections[$data['client_id']]->destroy();
                }
                break;
                // 廣播, Gateway::sendToAll($message, $client_id_array)
            case GatewayProtocol::CMD_SEND_TO_ALL:
                // $client_id_array不為空時，只廣播給$client_id_array指定的客戶端
                if($data['ext_data'])
                {
                    $client_id_array = unpack('N*', $data['ext_data']);
                    foreach($client_id_array as $client_id)
                    {
                        if(isset($this->_clientConnections[$client_id]))
                        {
                            $this->_clientConnections[$client_id]->send($data['body']);
                        }
                    }
                }
                // $client_id_array為空時，廣播給所有在線客戶端
                else
                {
                    foreach($this->_clientConnections as $client_connection)
                    {
                        $client_connection->send($data['body']);
                    }
                }
                break;
                // 更新客戶端session
            case GatewayProtocol::CMD_UPDATE_SESSION:
                if(isset($this->_clientConnections[$data['client_id']]))
                {
                    $this->_clientConnections[$data['client_id']]->session = $data['ext_data'];
                }
                break;
                // 獲得客戶端在線狀態 Gateway::getOnlineStatus()
            case GatewayProtocol::CMD_GET_ONLINE_STATUS:
                $online_status = json_encode(array_keys($this->_clientConnections));
                $connection->send($online_status);
                break;
                // 判斷某個client_id是否在線 Gateway::isOnline($client_id)
            case GatewayProtocol::CMD_IS_ONLINE:
                $connection->send((int)isset($this->_clientConnections[$data['client_id']]));
                break;
            default :
                $err_msg = "gateway inner pack err cmd=$cmd";
                throw new \Exception($err_msg);
        }
    }
    
    /**
     * 當worker連接關閉時
     * @param TcpConnection $connection
     */
    public function onWorkerClose($connection)
    {
        //$this->log("{$connection->remoteAddress} CLOSE INNER_CONNECTION\n");
        unset($this->_workerConnections[$connection->remoteAddress]);
    }
    
    /**
     * 存儲當前Gateway的內部通信地址
     * @param string $address
     * @return bool
     */
    protected function registerAddress()
    {
        $address = $this->lanIp.':'.$this->lanPort;
        // key
        $key = 'GLOBAL_GATEWAY_ADDRESS';
        try
        {
            $store = Store::instance('gateway');
        }
        catch(\Exception $msg)
        {
            $this->log($msg);
            return false;
        }
        // 為保證原子性，需要加鎖
        Lock::get();
        $addresses_list = $store->get($key);
        if(empty($addresses_list))
        {
            $addresses_list = array();
        }
        $addresses_list[$address] = $address;
        if(!$store->set($key, $addresses_list))
        {
            Lock::release();
            if(get_class($store) == 'Memcached')
            {
                $msg = " registerAddress fail : Memcache Error " . $store->getResultMessage();
            }
            $this->log($msg);
            return false;
        }
        Lock::release();
        return true;
    }
    
    /**
     * 刪除當前Gateway的內部通信地址
     * @param string $address
     * @return bool
     */
    protected function unregisterAddress()
    {
        $address = $this->lanIp.':'.$this->lanPort;
        $key = 'GLOBAL_GATEWAY_ADDRESS';
        try
        {
            $store = Store::instance('gateway');
        }
        catch (\Exception $msg)
        {
            $this->log($msg);
            return false;
        }
        // 為保證原子性，需要加鎖
        Lock::get();
        $addresses_list = $store->get($key);
        if(empty($addresses_list))
        {
            $addresses_list = array();
        }
        unset($addresses_list[$address]);
        if(!$store->set($key, $addresses_list))
        {
            Lock::release();
            $msg = "unregisterAddress fail";
            if(get_class($store) == 'Memcached')
            {
                $msg .= " reason:".$store->getResultMessage();
            }
            $this->log($msg);
            return;
        }
        Lock::release();
        return true;
    }
    
    /**
     * 心跳邏輯
     * @return void
     */
    public function ping()
    {
        // 遍歷所有客戶端連接
        foreach($this->_clientConnections as $connection)
        {
            // 上次發送的心跳還沒有回復次數大於限定值就斷開
            if($this->pingNotResponseLimit > 0 && $connection->pingNotResponseCount >= $this->pingNotResponseLimit)
            {
                $connection->destroy();
                continue;
            }
            // $connection->pingNotResponseCount為-1說明最近客戶端有發來消息，則不給客戶端發送心跳
            if($connection->pingNotResponseCount++ >= 0)
            {
                if($this->pingData)
                {
                    $connection->send($this->pingData);
                }
            }
        }
    }
    
    /**
     * 當gateway關閉時觸發，清理數據
     * @return void
     */
    public function onWorkerStop()
    {
        $this->unregisterAddress();
        foreach($this->_clientConnections as $connection)
        {
            $this->delClientAddress($connection->globalClientId);
        }
        // 嘗試觸發用戶設置的回調
        if($this->_onWorkerStop)
        {
            call_user_func($this->_onWorkerStop, $this);
        }
    }
}