<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace GatewayWorker\Lib;

/**
 * 數據發送相關
 */
use \Workerman\Protocols\GatewayProtocol;
use \GatewayWorker\Lib\Store;
use \GatewayWorker\Lib\Context;

class Gateway
{
    /**
     * gateway實例
     * @var object
     */
    protected static  $businessWorker = null;
    
   /**
    * 向所有客戶端(或者client_id_array指定的客戶端)廣播消息
    * @param string $message 向客戶端發送的消息
    * @param array $client_id_array 客戶端id數組
    */
   public static function sendToAll($message, $client_id_array = null)
   {
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = GatewayProtocol::CMD_SEND_TO_ALL;
       $gateway_data['body'] = $message;
       
       if($client_id_array)
       {
           $params = array_merge(array('N*'), $client_id_array);
           $gateway_data['ext_data'] = call_user_func_array('pack', $params);
       }
       elseif(empty($client_id_array) && is_array($client_id_array))
       {
           return;
       }
       
       // 如果有businessWorker實例，說明運行在workerman環境中，通過businessWorker中的長連接發送數據
       if(self::$businessWorker)
       {
           foreach(self::$businessWorker->gatewayConnections as $gateway_connection)
           {
               $gateway_connection->send($gateway_data);
           }
       }
       // 運行在其它環境中，使用udp向worker發送數據
       else
       {
           $all_addresses = Store::instance('gateway')->get('GLOBAL_GATEWAY_ADDRESS');
           if(!$all_addresses)
           {
               throw new \Exception('GLOBAL_GATEWAY_ADDRESS is ' . var_export($all_addresses, true));
           }
           foreach($all_addresses as $address)
           {
               self::sendToGateway($address, $gateway_data);
           }
       }
   }
   
   /**
    * 向某個客戶端發消息
    * @param int $client_id 
    * @param string $message
    */
   public static function sendToClient($client_id, $message)
   {
       return self::sendCmdAndMessageToClient($client_id, GatewayProtocol::CMD_SEND_TO_ONE, $message);
   } 
   
   /**
    * 向當前客戶端發送消息
    * @param string $message
    */
   public static function sendToCurrentClient($message)
   {
       return self::sendCmdAndMessageToClient(null, GatewayProtocol::CMD_SEND_TO_ONE, $message);
   }
   
   /**
    * 判斷某個客戶端是否在線
    * @param int $client_id
    * @return 0/1
    */
   public static function isOnline($client_id)
   {
       $address = Store::instance('gateway')->get('client_id-'.$client_id);
       if(!$address)
       {
           return 0;
       }
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = GatewayProtocol::CMD_IS_ONLINE;
       $gateway_data['client_id'] = $client_id;
       return self::sendUdpAndRecv($address, $gateway_data);
   }
   
   /**
    * 獲取在線狀態，目前返回一個在線client_id數組
    * @return array
    */
   public static function getOnlineStatus()
   {
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = GatewayProtocol::CMD_GET_ONLINE_STATUS;
       $gateway_buffer = GatewayProtocol::encode($gateway_data);
       
       $all_addresses = Store::instance('gateway')->get('GLOBAL_GATEWAY_ADDRESS');
       $client_array = $status_data = array();
       // 批量向所有gateway進程發送CMD_GET_ONLINE_STATUS命令
       foreach($all_addresses as $address)
       {
           $client = stream_socket_client("udp://$address", $errno, $errmsg);
           if(strlen($gateway_buffer) === stream_socket_sendto($client, $gateway_buffer))
           {
               $client_id = (int) $client;
               $client_array[$client_id] = $client;
           }
       }
       // 超時1秒
       $time_out = 1;
       $time_start = microtime(true);
       // 批量接收請求
       while(count($client_array) > 0)
       {
           $write = $except = array();
           $read = $client_array;
           if(@stream_select($read, $write, $except, $time_out))
           {
               foreach($read as $client)
               {
                   // udp
                   $data = json_decode(fread($client, 65535), true);
                   if($data)
                   {
                       $status_data = array_merge($status_data, $data);
                   }
                   unset($client_array[$client]);
               }
           }
           if(microtime(true) - $time_start > $time_out)
           {
               break;
           }
       }
       return $status_data;
   }
   
   /**
    * 關閉某個客戶端
    * @param int $client_id
    * @param string $message
    */
   public static function closeClient($client_id)
   {
       if($client_id === Context::$client_id)
       {
           return self::closeCurrentClient();
       }
       // 不是發給當前用戶則使用存儲中的地址
       else
       {
           $address = Store::instance('gateway')->get('client_id-'.$client_id);
           if(!$address)
           {
               return false;
           }
           return self::kickAddress($address, $client_id);
       }
   }
   
   /**
    * 踢掉當前客戶端
    * @param string $message
    */
   public static function closeCurrentClient()
   {
       return self::kickAddress(Context::$local_ip.':'.Context::$local_port, Context::$client_id);
   }
   
   /**
    * 更新session,框架自動調用，開發者不要調用
    * @param int $client_id
    * @param string $session_str
    */
   public static function updateSocketSession($client_id, $session_str)
   {
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = GatewayProtocol::CMD_UPDATE_SESSION;
       $gateway_data['client_id'] = $client_id;
       $gateway_data['ext_data'] = $session_str;
       return self::sendToGateway(Context::$local_ip . ':' . Context::$local_port, $gateway_data);
   }
   
   /**
    * 想某個用戶網關發送命令和消息
    * @param int $client_id
    * @param int $cmd
    * @param string $message
    * @return boolean
    */
   protected static function sendCmdAndMessageToClient($client_id, $cmd , $message)
   {
       // 如果是發給當前用戶則直接獲取上下文中的地址
       if($client_id === Context::$client_id || $client_id === null)
       {
           $address = Context::$local_ip.':'.Context::$local_port;
       }
       else
       {
           $address = Store::instance('gateway')->get('client_id-'.$client_id);
           if(!$address)
           {
               return false;
           }
       }
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = $cmd;
       $gateway_data['client_id'] = $client_id ? $client_id : Context::$client_id;
       $gateway_data['body'] = $message;
       
       return self::sendToGateway($address, $gateway_data);
   }
   
   /**
    * 發送udp數據並返回
    * @param int $address
    * @param string $message
    * @return boolean
    */
   protected static function sendUdpAndRecv($address , $data)
   {
       $buffer = GatewayProtocol::encode($data);
       // 非workerman環境，使用udp發送數據
       $client = stream_socket_client("udp://$address", $errno, $errmsg);
       if(strlen($buffer) == stream_socket_sendto($client, $buffer))
       {
           // 阻塞讀
           stream_set_blocking($client, 1);
           // 1秒超時
           stream_set_timeout($client, 1);
           // 讀udp數據
           $data = fread($client, 655350);
           // 返回結果
           return json_decode($data, true);
       }
       else
       {
           throw new \Exception("sendUdpAndRecv($address, \$bufer) fail ! Can not send UDP data!", 502);
       }
   }
   
   /**
    * 發送數據到網關
    * @param string $address
    * @param string $buffer
    */
   protected static function sendToGateway($address, $gateway_data)
   {
       // 有$businessWorker說明是workerman環境，使用$businessWorker發送數據
       if(self::$businessWorker)
       {
           if(!isset(self::$businessWorker->gatewayConnections[$address]))
           {
               return false;
           }
           return self::$businessWorker->gatewayConnections[$address]->send($gateway_data);
       }
       // 非workerman環境，使用udp發送數據
       $gateway_buffer = GatewayProtocol::encode($gateway_data);
       $client = stream_socket_client("udp://$address", $errno, $errmsg);
       return strlen($gateway_buffer) == stream_socket_sendto($client, $gateway_buffer);
   }
   
   /**
    * 踢掉某個網關的socket
    * @param string $local_ip
    * @param int $local_port
    * @param int $client_id
    * @param string $message
    * @param int $client_id
    */
   protected  static function kickAddress($address, $client_id)
   {
       $gateway_data = GatewayProtocol::$empty;
       $gateway_data['cmd'] = GatewayProtocol::CMD_KICK;
       $gateway_data['client_id'] = $client_id;
       return self::sendToGateway($address, $gateway_data);
   }
   
   /**
    * 設置gateway實例
    * @param Bootstrap/Gateway $gateway_instance
    */
   public static function setBusinessWorker($business_worker_instance)
   {
       self::$businessWorker = $business_worker_instance;
   }
 
}
