<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman\Connection;

use Workerman\Events\Libevent;
use Workerman\Events\Select;
use Workerman\Events\EventInterface;
use Workerman\Worker;
use \Exception;

/**
 * 異步tcp連接類 
 */
class AsyncTcpConnection extends TcpConnection
{
    
    /**
     * 當連接成功時，如果設置了連接成功回調，則執行
     * @var callback
     */
    public $onConnect = null;
    
    /**
     * 連接狀態 連接中
     * @var int
     */
    protected $_status = self::STATUS_CONNECTING;
    
    /**
     * 構造函數，創建連接
     * @param resource $socket
     * @param EventInterface $event
     */
    public function __construct($remote_address)
    {
        list($scheme, $address) = explode(':', $remote_address, 2);
        if($scheme != 'tcp')
        {
            // 判斷協議類是否存在
            $scheme = ucfirst($scheme);
            $this->protocol = '\\Protocols\\'.$scheme;
            if(!class_exists($this->protocol))
            {
                $this->protocol = '\\Workerman\\Protocols\\' . $scheme;
                if(!class_exists($this->protocol))
                {
                    throw new Exception("class \\Protocols\\$scheme not exist");
                }
            }
        }
        $this->_remoteAddress = substr($address, 2);
        $this->id = self::$_idRecorder++;
        // 統計數據
        self::$statistics['connection_count']++;
        $this->maxSendBufferSize = self::$defaultMaxSendBufferSize;
    }
    
    public function connect()
    {
        // 創建異步連接
        $this->_socket = stream_socket_client("tcp://{$this->_remoteAddress}", $errno, $errstr, 0, STREAM_CLIENT_ASYNC_CONNECT);
        // 如果失敗嘗試觸發失敗回調（如果有回調的話）
        if(!$this->_socket)
        {
            $this->_status = self::STATUS_CLOSED;
            $this->emitError(WORKERMAN_CONNECT_FAIL, $errstr);
            return;
        }
        // 監聽連接可寫事件（可寫意味著連接已經建立或者已經出錯）
        Worker::$globalEvent->add($this->_socket, EventInterface::EV_WRITE, array($this, 'checkConnection'));
    }
    
    /**
     * 嘗試觸發失敗回調
     * @param int $code
     * @param string $msg
     * @return void
     */
    protected function emitError($code, $msg)
    {
        if($this->onError)
        {
            try{
                call_user_func($this->onError, $this, $code, $msg);
            }
            catch(Exception $e)
            {
                echo $e;
            }
        }
    }
    
    /**
     * 檢查連接狀態，連接成功還是失敗
     * @param resource $socket
     * @return void
     */
    public function checkConnection($socket)
    {
        // 刪除連接可寫監聽
        Worker::$globalEvent->del($this->_socket, EventInterface::EV_WRITE);
        // 需要判斷兩次連接是否已經斷開
        if(!feof($this->_socket) && !feof($this->_socket))
        {
            // 設置非阻塞
            stream_set_blocking($this->_socket, 0);
            // 監聽可讀事件
            Worker::$globalEvent->add($this->_socket, EventInterface::EV_READ, array($this, 'baseRead'));
            // 如果發送緩衝區有數據則執行發送
            if($this->_sendBuffer)
            {
                Worker::$globalEvent->add($this->_socket, EventInterface::EV_WRITE, array($this, 'baseWrite'));
            }
            // 標記狀態為連接已經建立
            $this->_status = self::STATUS_ESTABLISH;
            // 如果有設置onConnect回調，則執行
            if($this->onConnect)
            {
                try 
                {
                    call_user_func($this->onConnect, $this);
                }
                catch(Exception $e)
                {
                    self::$statistics['throw_exception']++;
                    echo $e;
                }
            }
        }
        else
        {
            $this->_status = self::STATUS_CLOSED;
            // 關閉socket
            @fclose($this->_socket);
            // 連接未建立成功
            $this->emitError(WORKERMAN_CONNECT_FAIL, 'connect fail');
        }
    }
}
