<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman\Connection;

use Workerman\Events\Libevent;
use Workerman\Events\Select;
use Workerman\Events\EventInterface;
use Workerman\Worker;
use \Exception;

/**
 * Tcp連接類 
 */
class TcpConnection extends ConnectionInterface
{
    /**
     * 當數據可讀時，從socket緩衝區讀取多少字節數據
     * @var int
     */
    const READ_BUFFER_SIZE = 8192;

    /**
     * 連接狀態 連接中
     * @var int
     */
    const STATUS_CONNECTING = 1;
    
    /**
     * 連接狀態 已經建立連接
     * @var int
     */
    const STATUS_ESTABLISH = 2;

    /**
     * 連接狀態 連接關閉中，標識調用了close方法，但是發送緩衝區中任然有數據
     * 等待發送緩衝區的數據發送完畢（寫入到socket寫緩衝區）後執行關閉
     * @var int
     */
    const STATUS_CLOSING = 4;
    
    /**
     * 連接狀態 已經關閉
     * @var int
     */
    const STATUS_CLOSED = 8;
    
    /**
     * 當對端發來數據時，如果設置了$onMessage回調，則執行
     * @var callback
     */
    public $onMessage = null;
    
    /**
     * 當連接關閉時，如果設置了$onClose回調，則執行
     * @var callback
     */
    public $onClose = null;
    
    /**
     * 當出現錯誤是，如果設置了$onError回調，則執行
     * @var callback
     */
    public $onError = null;
    
    /**
     * 當發送緩衝區滿時，如果設置了$onBufferFull回調，則執行
     * @var callback
     */
    public $onBufferFull = null;
    
    /**
     * 當發送緩衝區被清空時，如果設置了$onBufferDrain回調，則執行
     * @var callback
     */
    public $onBufferDrain = null;
    
    /**
     * 使用的應用層協議，是協議類的名稱
     * 值類似於 Workerman\\Protocols\\Http
     * @var string
     */
    public $protocol = '';
    
    /**
     * 屬於哪個worker
     * @var Worker
     */
    public $worker = null;
    
    /**
     * 連接的id，一個自增整數
     * @var int
     */
    public $id = 0;
    
    /**
     * 設置當前連接的最大發送緩衝區大小，默認大小為TcpConnection::$defaultMaxSendBufferSize
     * 當發送緩衝區滿時，會嘗試觸發onBufferFull回調（如果有設置的話）
     * 如果沒設置onBufferFull回調，由於發送緩衝區滿，則後續發送的數據將被丟棄，
     * 並觸發onError回調，直到發送緩衝區有空位
     * 注意 此值可以動態設置
     * @var int
     */
    public $maxSendBufferSize = 1048576;
    
    /**
     * 默認發送緩衝區大小，設置此屬性會影響所有連接的默認發送緩衝區大小
     * 如果想設置某個連接發送緩衝區的大小，可以單獨設置對應連接的$maxSendBufferSize屬性
     * @var int
     */
    public static $defaultMaxSendBufferSize = 1048576;
    
    /**
     * 能接受的最大數據包，為了防止惡意攻擊，當數據包的大小大於此值時執行斷開
     * 注意 此值可以動態設置
     * 例如 Workerman\Connection\TcpConnection::$maxPackageSize=1024000;
     * @var int
     */
    public static $maxPackageSize = 10485760;
    
    /**
     * id 記錄器
     * @var int
     */
    protected static $_idRecorder = 1;
    
    /**
     * 實際的socket資源
     * @var resource
     */
    protected $_socket = null;

    /**
     * 發送緩衝區
     * @var string
     */
    protected $_sendBuffer = '';
    
    /**
     * 接收緩衝區
     * @var string
     */
    protected $_recvBuffer = '';
    
    /**
     * 當前正在處理的數據包的包長（此值是協議的intput方法的返回值）
     * @var int
     */
    protected $_currentPackageLength = 0;
    
    /**
     * 當前的連接狀態
     * @var int
     */
    protected $_status = self::STATUS_ESTABLISH;
    
    /**
     * 對端ip
     * @var string
     */
    protected $_remoteIp = '';
    
    /**
     * 對端端口
     * @var int
     */
    protected $_remotePort = 0;
    
    /**
     * 對端的地址 ip+port
     * 值類似於 192.168.1.100:3698
     * @var string
     */
    protected $_remoteAddress = '';
    
    /**
     * 是否是停止接收數據
     * @var bool
     */
    protected $_isPaused = false;
    
    /**
     * 構造函數
     * @param resource $socket
     * @param EventInterface $event
     */
    public function __construct($socket)
    {
        // 統計數據
        self::$statistics['connection_count']++;
        $this->id = self::$_idRecorder++;
        $this->_socket = $socket;
        stream_set_blocking($this->_socket, 0);
        Worker::$globalEvent->add($this->_socket, EventInterface::EV_READ, array($this, 'baseRead'));
        $this->maxSendBufferSize = self::$defaultMaxSendBufferSize;
    }
    
    /**
     * 發送數據給對端
     * @param string $send_buffer
     * @param bool $raw
     * @return void|boolean
     */
    public function send($send_buffer, $raw = false)
    {
        // 如果沒有設置以原始數據發送，並且有設置協議則按照協議編碼
        if(false === $raw && $this->protocol)
        {
            $parser = $this->protocol;
            $send_buffer = $parser::encode($send_buffer, $this);
        }
        
        // 如果當前狀態是連接中，則把數據放入發送緩衝區
        if($this->_status === self::STATUS_CONNECTING)
        {
            $this->_sendBuffer .= $send_buffer;
            return null;
        }
        // 如果當前連接是關閉，則返回false
        elseif($this->_status === self::STATUS_CLOSING || $this->_status === self::STATUS_CLOSED)
        {
            return false;
        }
        
        // 如果發送緩衝區為空，嘗試直接發送
        if($this->_sendBuffer === '')
        {
            // 直接發送
            $len = @fwrite($this->_socket, $send_buffer);
            // 所有數據都發送完畢
            if($len === strlen($send_buffer))
            {
                return true;
            }
            // 只有部分數據發送成功
            if($len > 0)
            {
                // 未發送成功部分放入發送緩衝區
                $this->_sendBuffer = substr($send_buffer, $len);
            }
            else
            {
                // 如果連接斷開
                if(feof($this->_socket))
                {
                    // status統計發送失敗次數
                    self::$statistics['send_fail']++;
                    // 如果有設置失敗回調，則執行
                    if($this->onError)
                    {
                        try
                        {
                            call_user_func($this->onError, $this, WORKERMAN_SEND_FAIL, 'client closed');
                        }
                        catch(Exception $e)
                        {
                            echo $e;
                        }
                    }
                    // 銷毀連接
                    $this->destroy();
                    return false;
                }
                // 連接未斷開，發送失敗，則把所有數據放入發送緩衝區
                $this->_sendBuffer = $send_buffer;
            }
            // 監聽對端可寫事件
            Worker::$globalEvent->add($this->_socket, EventInterface::EV_WRITE, array($this, 'baseWrite'));
            // 檢查發送緩衝區是否已滿，如果滿了嘗試觸發onBufferFull回調
            $this->checkBufferIsFull();
            return null;
        }
        else
        {
            // 緩衝區已經標記為滿，仍然然有數據發送，則丟棄數據包
            if($this->maxSendBufferSize <= strlen($this->_sendBuffer))
            {
                // 為status命令統計發送失敗次數
                self::$statistics['send_fail']++;
                // 如果有設置失敗回調，則執行
                if($this->onError)
                {
                    try
                    {
                        call_user_func($this->onError, $this, WORKERMAN_SEND_FAIL, 'send buffer full and drop package');
                    }
                    catch(Exception $e)
                    {
                        echo $e;
                    }
                }
                return false;
            }
            // 將數據放入放緩衝區
            $this->_sendBuffer .= $send_buffer;
            // 檢查發送緩衝區是否已滿，如果滿了嘗試觸發onBufferFull回調
            $this->checkBufferIsFull();
        }
    }
    
    /**
     * 獲得對端ip
     * @return string
     */
    public function getRemoteIp()
    {
        if(!$this->_remoteIp)
        {
            $this->_remoteAddress = stream_socket_get_name($this->_socket, true);
            if($this->_remoteAddress)
            {
                list($this->_remoteIp, $this->_remotePort) = explode(':', $this->_remoteAddress, 2);
                $this->_remotePort = (int)$this->_remotePort;
            }
        }
        return $this->_remoteIp;
    }
    
    /**
     * 獲得對端端口
     * @return int
     */
    public function getRemotePort()
    {
        if(!$this->_remotePort)
        {
            $this->_remoteAddress = stream_socket_get_name($this->_socket, true);
            if($this->_remoteAddress)
            {
                list($this->_remoteIp, $this->_remotePort) = explode(':', $this->_remoteAddress, 2);
                $this->_remotePort = (int)$this->_remotePort;
            }
        }
        return $this->_remotePort;
    }
    
    /**
     * 暫停接收數據，一般用於控制上傳流量
     * @return void
     */
    public function pauseRecv()
    {
        Worker::$globalEvent->del($this->_socket, EventInterface::EV_READ);
        $this->_isPaused = true;
    }
    
    /**
     * 恢復接收數據，一般用戶控制上傳流量
     * @return void
     */
    public function resumeRecv()
    {
        if($this->_isPaused === true)
        {
            Worker::$globalEvent->add($this->_socket, EventInterface::EV_READ, array($this, 'baseRead'));
            $this->_isPaused = false;
            $this->baseRead($this->_socket);
        }
    }

    /**
     * 當socket可讀時的回調
     * @param resource $socket
     * @return void
     */
    public function baseRead($socket)
    {
        if(!is_resource($socket) || feof($socket))
        {
            $this->destroy();
            return;
        }
       while(1)
       {
           $buffer = fread($socket, self::READ_BUFFER_SIZE);
           if($buffer === '' || $buffer === false)
           {
               break;
           }
           $this->_recvBuffer .= $buffer; 
       }
       
       if($this->_recvBuffer)
       {
           if(!$this->onMessage)
           {
               $this->_recvBuffer = '';
               return ;
           }
           
           // 如果設置了協議
           if($this->protocol)
           {
               $parser = $this->protocol;
               while($this->_recvBuffer && !$this->_isPaused)
               {
                   // 當前包的長度已知
                   if($this->_currentPackageLength)
                   {
                       // 數據不夠一個包
                       if($this->_currentPackageLength > strlen($this->_recvBuffer))
                       {
                           break;
                       }
                   }
                   else
                   {
                       // 獲得當前包長
                       $this->_currentPackageLength = $parser::input($this->_recvBuffer, $this);
                       // 數據不夠，無法獲得包長
                       if($this->_currentPackageLength === 0)
                       {
                           break;
                       }
                       elseif($this->_currentPackageLength > 0 && $this->_currentPackageLength <= self::$maxPackageSize)
                       {
                           // 數據不夠一個包
                           if($this->_currentPackageLength > strlen($this->_recvBuffer))
                           {
                               break;
                           }
                       }
                       // 包錯誤
                       else
                       {
                           $this->close('error package. package_length='.var_export($this->_currentPackageLength, true));
                           return;
                       }
                   }
                   
                   // 數據足夠一個包長
                   self::$statistics['total_request']++;
                   // 當前包長剛好等於buffer的長度
                   if(strlen($this->_recvBuffer) === $this->_currentPackageLength)
                   {
                       $one_request_buffer = $this->_recvBuffer;
                       $this->_recvBuffer = '';
                   }
                   else
                   {
                       // 從緩衝區中獲取一個完整的包
                       $one_request_buffer = substr($this->_recvBuffer, 0, $this->_currentPackageLength);
                       // 將當前包從接受緩衝區中去掉
                       $this->_recvBuffer = substr($this->_recvBuffer, $this->_currentPackageLength);
                   }
                   // 重置當前包長為0
                   $this->_currentPackageLength = 0;
                   // 處理數據包
                   try
                   {
                       call_user_func($this->onMessage, $this, $parser::decode($one_request_buffer, $this));
                   }
                   catch(Exception $e)
                   {
                       self::$statistics['throw_exception']++;
                       echo $e;
                   }
               }
               return;
           }
           // 沒有設置協議，則直接把接收的數據當做一個包處理
           self::$statistics['total_request']++;
           try 
           {
               call_user_func($this->onMessage, $this, $this->_recvBuffer);
           }
           catch(Exception $e)
           {
               self::$statistics['throw_exception']++;
               echo $e;
           }
           // 清空緩衝區
           $this->_recvBuffer = '';
       }
    }

    /**
     * socket可寫時的回調
     * @return void
     */
    public function baseWrite()
    {
        $len = @fwrite($this->_socket, $this->_sendBuffer);
        if($len === strlen($this->_sendBuffer))
        {
            Worker::$globalEvent->del($this->_socket, EventInterface::EV_WRITE);
            $this->_sendBuffer = '';
            // 發送緩衝區的數據被發送完畢，嘗試觸發onBufferDrain回調
            if($this->onBufferDrain)
            {
                try 
                {
                    call_user_func($this->onBufferDrain, $this);
                }
                catch(Exception $e)
                {
                    echo $e;
                }
            }
            // 如果連接狀態為關閉，則銷毀連接
            if($this->_status === self::STATUS_CLOSING)
            {
                $this->destroy();
            }
            return true;
        }
        if($len > 0)
        {
           $this->_sendBuffer = substr($this->_sendBuffer, $len);
        }
        else
        {
           if(feof($this->_socket))
           {
               self::$statistics['send_fail']++;
               $this->destroy();
           }
        }
    }
    
    /**
     * 從緩衝區中消費掉$length長度的數據
     * @param int $length
     * @return void
     */
    public function consumeRecvBuffer($length)
    {
        $this->_recvBuffer = substr($this->_recvBuffer, $length);
    }

    /**
     * 關閉連接
     * @param mixed $data
     * @void
     */
    public function close($data = null)
    {
        if($this->_status === self::STATUS_CLOSING || $this->_status === self::STATUS_CLOSED)
        {
            return false;
        }
        else
        {
            if($data !== null)
            {
                $this->send($data);
            }
            $this->_status = self::STATUS_CLOSING;
        }
        if($this->_sendBuffer === '')
        {
           $this->destroy();
        }
    }
    
    /**
     * 獲得socket連接
     * @return resource
     */
    public function getSocket()
    {
        return $this->_socket;
    }

    /**
     * 檢查發送緩衝區是否已滿，如果滿了嘗試觸發onBufferFull回調
     * @return void
     */
    protected function checkBufferIsFull()
    {
        if($this->maxSendBufferSize <= strlen($this->_sendBuffer))
        {
            if($this->onBufferFull)
            {
                try
                {
                    call_user_func($this->onBufferFull, $this);
                }
                catch(Exception $e)
                {
                    echo $e;
                }
            }
        }
    }
    /**
     * 銷毀連接
     * @return void
     */
    public function destroy()
    {
        // 避免重複調用
        if($this->_status === self::STATUS_CLOSED)
        {
            return false;
        }
        // 刪除事件監聽
        Worker::$globalEvent->del($this->_socket, EventInterface::EV_READ);
        Worker::$globalEvent->del($this->_socket, EventInterface::EV_WRITE);
        // 關閉socket
        @fclose($this->_socket);
        // 從連接中刪除
        if($this->worker)
        {
            unset($this->worker->connections[(int)$this->_socket]);
        }
        // 標記該連接已經關閉
       $this->_status = self::STATUS_CLOSED;
       // 觸發onClose回調
       if($this->onClose)
       {
           try
           {
               call_user_func($this->onClose, $this);
           }
           catch (Exception $e)
           {
               self::$statistics['throw_exception']++;
               echo $e;
           }
       }
    }
    
    /**
     * 析構函數
     * @return void
     */
    public function __destruct()
    {
        // 統計數據
        self::$statistics['connection_count']--;
    }
}
