<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman\Connection;

use Workerman\Events\Libevent;
use Workerman\Events\Select;
use Workerman\Events\EventInterface;
use Workerman\Worker;
use \Exception;

/**
 * udp連接類（udp實際上是無連接的，這裡是為了保持與TCP接口一致） 
 */
class UdpConnection extends ConnectionInterface
{
    /**
     * 應用層協議
     * 值類似於 Workerman\\Protocols\\Http
     * @var string
     */
    public $protocol = '';
    
    /**
     * udp socket 資源
     * @var resource
     */
    protected $_socket = null;
    
    /**
     * 對端 ip
     * @var string
     */
    protected $_remoteIp = '';
    
    /**
     * 對端 端口
     * @var int
     */
    protected $_remotePort = 0;
    
    /**
     * 對端 地址
     * 值類似於 192.168.10.100:3698
     * @var string
     */
    protected $_remoteAddress = '';

    /**
     * 構造函數
     * @param resource $socket
     * @param string $remote_address
     */
    public function __construct($socket, $remote_address)
    {
        $this->_socket = $socket;
        $this->_remoteAddress = $remote_address;
    }
    
    /**
     * 發送數據給對端
     * @param string $send_buffer
     * @return void|boolean
     */
    public function send($send_buffer)
    {
        return strlen($send_buffer) === stream_socket_sendto($this->_socket, $send_buffer, 0, $this->_remoteAddress);
    }
    
    /**
     * 獲得對端 ip
     * @return string
     */
    public function getRemoteIp()
    {
        if(!$this->_remoteIp)
        {
            list($this->_remoteIp, $this->_remotePort) = explode(':', $this->_remoteAddress, 2);
        }
        return $this->_remoteIp;
    }
    
    /**
     * 獲得對端端口
     */
    public function getRemotePort()
    {
        if(!$this->_remotePort)
        {
            list($this->_remoteIp, $this->_remotePort) = explode(':', $this->_remoteAddress, 2);
        }
        return $this->_remotePort;
    }

    /**
     * 關閉連接（此處為了保持與TCP接口一致，提供了close方法）
     * @void
     */
    public function close($data = null)
    {
        if($data !== null)
        {
            $this->send($data);
        }
        return true;
    }
}
