<?php 
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman\Protocols;

/**
 * Gateway與Worker間通訊的二進制協議
 * 
 * struct GatewayProtocol
 * {
 *     unsigned int        pack_len,
 *     unsigned char     cmd,//命令字
 *     unsigned int        local_ip,
 *     unsigned short    local_port,
 *     unsigned int        client_ip,
 *     unsigned short    client_port,
 *     unsigned int        client_id,
 *     unsigned char      flag,
 *     unsigned int        ext_len,
 *     char[ext_len]        ext_data,
 *     char[pack_length-HEAD_LEN] body//包體
 * }
 * 
 */
class GatewayProtocol
{
    // 發給worker，gateway有一個新的連接
    const CMD_ON_CONNECTION = 1;
    
    // 發給worker的，客戶端有消息
    const CMD_ON_MESSAGE = 3;
    
    // 發給worker上的關閉鏈接事件
    const CMD_ON_CLOSE = 4;
    
    // 發給gateway的向單個用戶發送數據
    const CMD_SEND_TO_ONE = 5;
    
    // 發給gateway的向所有用戶發送數據
    const CMD_SEND_TO_ALL = 6;
    
    // 發給gateway的踢出用戶
    const CMD_KICK = 7;
    
    // 發給gateway，通知用戶session更改
    const CMD_UPDATE_SESSION = 9;
    
    // 獲取在線狀態
    const CMD_GET_ONLINE_STATUS = 10;
    
    // 判斷是否在線
    const CMD_IS_ONLINE = 11;
    
    // 包體是標量
    const FLAG_BODY_IS_SCALAR = 0x01;
    
    /**
     * 包頭長度
     * @var integer
     */
    const HEAD_LEN = 26;
    
    public static $empty = array(
        'cmd' => 0,
        'local_ip' => '0.0.0.0',
        'local_port' => 0,
        'client_ip' => '0.0.0.0',
        'client_port' => 0,
        'client_id' => 0,
        'flag' => 0,
        'ext_data' => '',
        'body' => '',
    );
     
    /**
     * 返回包長度
     * @param string $buffer
     * @return int return current package length
     */
    public static function input($buffer)
    {
        if(strlen($buffer) < self::HEAD_LEN)
        {
            return 0;
        }
        
        $data = unpack("Npack_len", $buffer);
        return $data['pack_len'];
    }
    
    /**
     * 獲取整個包的buffer
     * @param array $data
     * @return string
     */
    public static function encode($data)
    {
        $flag = (int)is_scalar($data['body']);
        if(!$flag)
        {
            $data['body'] = serialize($data['body']);
        }
        $ext_len = strlen($data['ext_data']);
        $package_len = self::HEAD_LEN + $ext_len + strlen($data['body']);
        return pack("NCNnNnNNC",  $package_len,
                        $data['cmd'], ip2long($data['local_ip']), 
                        $data['local_port'], ip2long($data['client_ip']), 
                        $data['client_port'], $data['client_id'],
                       $ext_len, $flag) . $data['ext_data'] . $data['body'];
    }
    
    /**
     * 從二進制數據轉換為數組
     * @param string $buffer
     * @return array
     */    
    public static function decode($buffer)
    {
        $data = unpack("Npack_len/Ccmd/Nlocal_ip/nlocal_port/Nclient_ip/nclient_port/Nclient_id/Next_len/Cflag", $buffer);
        $data['local_ip'] = long2ip($data['local_ip']);
        $data['client_ip'] = long2ip($data['client_ip']);
        if($data['ext_len'] > 0)
        {
            $data['ext_data'] = substr($buffer, self::HEAD_LEN, $data['ext_len']);
            if($data['flag'] & self::FLAG_BODY_IS_SCALAR)
            {
                $data['body'] = substr($buffer, self::HEAD_LEN + $data['ext_len']);
            }
            else
            {
                $data['body'] = unserialize(substr($buffer, self::HEAD_LEN + $data['ext_len']));
            }
        }
        else
        {
            $data['ext_data'] = '';
            if($data['flag'] & self::FLAG_BODY_IS_SCALAR)
            {
                $data['body'] = substr($buffer, self::HEAD_LEN);
            }
            else
            {
                $data['body'] = unserialize(substr($buffer, self::HEAD_LEN));
            }
        }
        return $data;
    }
}



