<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman;

use \Workerman\Events\Libevent;
use \Workerman\Events\Select;
use \Workerman\Events\EventInterface;
use \Workerman\Connection\ConnectionInterface;
use \Workerman\Connection\TcpConnection;
use \Workerman\Connection\UdpConnection;
use \Workerman\Lib\Timer;
use \Workerman\Autoloader;
use \Exception;

/**
 * Worker 類
 * 是一個容器，用於監聽端口，維持客戶端連接
 */
class Worker
{
    /**
     * 版本號
     * @var string
     */
    const VERSION = '3.1.6';
    
    /**
     * 狀態 啟動中
     * @var int
     */
    const STATUS_STARTING = 1;
    
    /**
     * 狀態 運行中
     * @var int
     */
    const STATUS_RUNNING = 2;
    
    /**
     * 狀態 停止
     * @var int
     */
    const STATUS_SHUTDOWN = 4;
    
    /**
     * 狀態 平滑重啟中
     * @var int
     */
    const STATUS_RELOADING = 8;
    
    /**
     * 給子進程發送重啟命令 KILL_WORKER_TIMER_TIME 秒後
     * 如果對應進程仍然未重啟則強行殺死
     * @var int
     */
    const KILL_WORKER_TIMER_TIME = 1;
    
    /**
     * 默認的backlog，即內核中用於存放未被進程認領（accept）的連接隊列長度
     * @var int
     */
    const DEFAUL_BACKLOG = 1024;
    
    /**
     * udp最大包長
     * @var int
     */
    const MAX_UDP_PACKEG_SIZE = 65535;
    
    /**
     * worker的名稱，用於在運行status命令時標記進程
     * @var string
     */
    public $name = 'none';
    
    /**
     * 設置當前worker實例的進程數
     * @var int
     */
    public $count = 1;
    
    /**
     * 設置當前worker進程的運行用戶，啟動時需要root超級權限
     * @var string
     */
    public $user = '';
    
    /**
     * 當前worker進程是否可以平滑重啟 
     * @var bool
     */
    public $reloadable = true;
    
    /**
     * 當worker進程啟動時，如果設置了$onWorkerStart回調函數，則運行
     * 此鉤子函數一般用於進程啟動後初始化工作
     * @var callback
     */
    public $onWorkerStart = null;
    
    /**
     * 當有客戶端連接時，如果設置了$onConnect回調函數，則運行
     * @var callback
     */
    public $onConnect = null;
    
    /**
     * 當客戶端連接上發來數據時，如果設置了$onMessage回調，則運行
     * @var callback
     */
    public $onMessage = null;
    
    /**
     * 當客戶端的連接關閉時，如果設置了$onClose回調，則運行
     * @var callback
     */
    public $onClose = null;
    
    /**
     * 當客戶端的連接發生錯誤時，如果設置了$onError回調，則運行
     * 錯誤一般為客戶端斷開連接導致數據發送失敗、服務端的發送緩衝區滿導致發送失敗等
     * 具體錯誤碼及錯誤詳情會以參數的形式傳遞給回調，參見手冊
     * @var callback
     */
    public $onError = null;
    
    /**
     * 當連接的發送緩衝區滿時，如果設置了$onBufferFull回調，則執行
     * @var callback
     */
    public $onBufferFull = null;
    
    /**
     * 當鏈接的發送緩衝區被清空時，如果設置了$onBufferDrain回調，則執行
     * @var callback
     */
    public $onBufferDrain = null;
    
    /**
     * 當前進程退出時（由於平滑重啟或者服務停止導致），如果設置了此回調，則運行
     * @var callback
     */
    public $onWorkerStop = null;
    
    /**
     * 傳輸層協議
     * @var string
     */
    public $transport = 'tcp';
    
    /**
     * 所有的客戶端連接
     * @var array
     */
    public $connections = array();
    
    /**
     * 應用層協議，由初始化worker時指定
     * 例如 new worker('http://0.0.0.0:8080');指定使用http協議
     * @var string
     */
    protected $_protocol = '';
    
    /**
     * 當前worker實例初始化目錄位置，用於設置應用自動加載的根目錄
     * @var string
     */
    protected $_appInitPath = '';
    
    /**
     * 是否以守護進程的方式運行。運行start時加上-d參數會自動以守護進程方式運行
     * 例如 php start.php start -d
     * @var bool
     */
    public static $daemonize = false;
    
    /**
     * 重定向標準輸出，即將所有echo、var_dump等終端輸出寫到對應文件中
     * 注意 此參數只有在以守護進程方式運行時有效
     * @var string
     */
    public static $stdoutFile = '/dev/null';
    
    /**
     * pid文件的路徑及名稱
     * 例如 Worker::$pidFile = '/tmp/workerman.pid';
     * 注意 此屬性一般不必手動設置，默認會放到php臨時目錄中
     * @var string
     */
    public static $pidFile = '';
    
    /**
     * 日誌目錄，默認在workerman根目錄下，與Applications同級
     * 可以手動設置
     * 例如 Worker::$logFile = '/tmp/workerman.log';
     * @var unknown_type
     */
    public static $logFile = '';
    
    /**
     * 全局事件輪詢庫，用於監聽所有資源的可讀可寫事件
     * @var Select/Libevent
     */
    public static $globalEvent = null;
    
    /**
     * 主進程pid
     * @var int
     */
    protected static $_masterPid = 0;
    
    /**
     * 監聽的socket
     * @var stream
     */
    protected $_mainSocket = null;
    
    /**
     * socket名稱，包括應用層協議+ip+端口號，在初始化worker時設置 
     * 值類似 http://0.0.0.0:80
     * @var string
     */
    protected $_socketName = '';
    
    /**
     * socket的上下文，具體選項設置可以在初始化worker時傳遞
     * @var context
     */
    protected $_context = null;
    
    /**
     * 所有的worker實例
     * @var array
     */
    protected static $_workers = array();
    
    /**
     * 所有worker進程的pid
     * 格式為 [worker_id=>[pid=>pid, pid=>pid, ..], ..]
     * @var array
     */
    protected static $_pidMap = array();
    
    /**
     * 所有需要重啟的進程pid
     * 格式為 [pid=>pid, pid=>pid]
     * @var array
     */
    protected static $_pidsToRestart = array();
    
    /**
     * 當前worker狀態
     * @var int
     */
    protected static $_status = self::STATUS_STARTING;
    
    /**
     * 所有worke名稱(name屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxWorkerNameLength = 12;
    
    /**
     * 所有socket名稱(_socketName屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxSocketNameLength = 12;
    
    /**
     * 所有user名稱(user屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxUserNameLength = 12;
    
    /**
     * 運行 status 命令時用於保存結果的文件名
     * @var string
     */
    protected static $_statisticsFile = '';
    
    /**
     * 啟動的全局入口文件
     * 例如 php start.php start ，則入口文件為start.php
     * @var string
     */
    protected static $_startFile = '';
    
    /**
     * 全局統計數據，用於在運行 status 命令時展示
     * 統計的內容包括 workerman啟動的時間戳及每組worker進程的退出次數及退出狀態碼
     * @var array
     */
    protected static $_globalStatistics = array(
        'start_timestamp' => 0,
        'worker_exit_info' => array()
    );
    
    /**
     * 運行所有worker實例
     * @return void
     */
    public static function runAll()
    {
        // 初始化環境變量
        self::init();
        // 解析命令
        self::parseCommand();
        // 嘗試以守護進程模式運行
        self::daemonize();
        // 初始化所有worker實例，主要是監聽端口
        self::initWorkers();
        //  初始化所有信號處理函數
        self::installSignal();
        // 展示啟動界面
        self::displayUI();
        // 嘗試重定向標準輸入輸出
        self::resetStd();
        // 保存主進程pid
        self::saveMasterPid();
        // 創建子進程（worker進程）並運行
        self::forkWorkers();
        // 監控所有子進程（worker進程）
        self::monitorWorkers();
    }
    
    /**
     * 初始化一些環境變量
     * @return void
     */
    public static function init()
    {
        // 如果沒設置$pidFile，則生成默認值
        if(empty(self::$pidFile))
        {
            $backtrace = debug_backtrace();
            self::$_startFile = $backtrace[count($backtrace)-1]['file'];
            self::$pidFile = sys_get_temp_dir()."/workerman.".str_replace('/', '_', self::$_startFile).".pid";
        }
        // 沒有設置日誌文件，則生成一個默認值
        if(empty(self::$logFile))
        {
            self::$logFile = __DIR__ . '/../workerman.log';
        }
        // 標記狀態為啟動中
        self::$_status = self::STATUS_STARTING;
        // 啟動時間戳
        self::$_globalStatistics['start_timestamp'] = time();
        // 設置status文件位置
        self::$_statisticsFile = sys_get_temp_dir().'/workerman.status';
        // 嘗試設置進程名稱（需要php>=5.5或者安裝了proctitle擴展）
        self::setProcessTitle('WorkerMan: master process  start_file=' . self::$_startFile);
        
        // 初始化定時器
        Timer::init();
    }
    
    /**
     * 初始化所有的worker實例，主要工作為獲得格式化所需數據及監聽端口
     * @return void
     */
    protected static function initWorkers()
    {
        foreach(self::$_workers as $worker)
        {
            // 沒有設置worker名稱，則使用none代替
            if(empty($worker->name))
            {
                $worker->name = 'none';
            }
            // 獲得所有worker名稱中最大長度
            $worker_name_length = strlen($worker->name);
            if(self::$_maxWorkerNameLength < $worker_name_length)
            {
                self::$_maxWorkerNameLength = $worker_name_length;
            }
            // 獲得所有_socketName中最大長度
            $socket_name_length = strlen($worker->getSocketName());
            if(self::$_maxSocketNameLength < $socket_name_length)
            {
                self::$_maxSocketNameLength = $socket_name_length;
            }
            // 獲得運行用戶名的最大長度
            if(empty($worker->user) || posix_getuid() !== 0)
            {
                $worker->user = self::getCurrentUser();
            }
            $user_name_length = strlen($worker->user);
            if(self::$_maxUserNameLength < $user_name_length)
            {
                self::$_maxUserNameLength = $user_name_length;
            }
            // 監聽端口
            $worker->listen();
        }
    }
    
    /**
     * 獲得運行當前進程的用戶名
     * @return string
     */
    protected static function getCurrentUser()
    {
        $user_info = posix_getpwuid(posix_getuid());
        return $user_info['name'];
    }
    
    /**
     * 展示啟動界面
     * @return void
     */
    protected static function displayUI()
    {
        echo "\033[1A\n\033[K-----------------------\033[47;30m WORKERMAN \033[0m-----------------------------\n\033[0m";
        echo 'Workerman version:' , Worker::VERSION , "          PHP version:",PHP_VERSION,"\n";
        echo "------------------------\033[47;30m WORKERS \033[0m-------------------------------\n";
        echo "\033[47;30muser\033[0m",str_pad('', self::$_maxUserNameLength+2-strlen('user')), "\033[47;30mworker\033[0m",str_pad('', self::$_maxWorkerNameLength+2-strlen('worker')), "\033[47;30mlisten\033[0m",str_pad('', self::$_maxSocketNameLength+2-strlen('listen')), "\033[47;30mprocesses\033[0m \033[47;30m","status\033[0m\n";
        foreach(self::$_workers as $worker)
        {
            echo str_pad($worker->user, self::$_maxUserNameLength+2),str_pad($worker->name, self::$_maxWorkerNameLength+2),str_pad($worker->getSocketName(), self::$_maxSocketNameLength+2), str_pad(' '.$worker->count, 9), " \033[32;40m [OK] \033[0m\n";;
        }
        echo "----------------------------------------------------------------\n";
    }
    
    /**
     * 解析運行命令
     * php yourfile.php start | stop | restart | reload | status
     * @return void
     */
    public static function parseCommand()
    {
        // 檢查運行命令的參數
        global $argv;
        $start_file = $argv[0]; 
        if(!isset($argv[1]))
        {
            exit("Usage: php yourfile.php {start|stop|restart|reload|status}\n");
        }
        
        // 命令
        $command = trim($argv[1]);
        
        // 子命令，目前只支持-d
        $command2 = isset($argv[2]) ? $argv[2] : '';
        
        // 記錄日誌
        self::log("Workerman[$start_file] $command");
        
        // 檢查主進程是否在運行
        $master_pid = @file_get_contents(self::$pidFile);
        $master_is_alive = $master_pid && @posix_kill($master_pid, 0);
        if($master_is_alive)
        {
            if($command === 'start')
            {
                self::log("Workerman[$start_file] is running");
            }
        }
        elseif($command !== 'start' && $command !== 'restart')
        {
            self::log("Workerman[$start_file] not run");
        }
        
        // 根據命令做相應處理
        switch($command)
        {
            // 啟動 workerman
            case 'start':
                if($command2 === '-d')
                {
                    Worker::$daemonize = true;
                }
                break;
            // 顯示 workerman 運行狀態
            case 'status':
                // 嘗試刪除統計文件，避免髒數據
                if(is_file(self::$_statisticsFile))
                {
                    @unlink(self::$_statisticsFile);
                }
                // 向主進程發送 SIGUSR2 信號 ，然後主進程會向所有子進程發送 SIGUSR2 信號
                // 所有進程收到 SIGUSR2 信號後會向 $_statisticsFile 寫入自己的狀態
                posix_kill($master_pid, SIGUSR2);
                // 睡眠100毫秒，等待子進程將自己的狀態寫入$_statisticsFile指定的文件
                usleep(100000);
                // 展示狀態
                readfile(self::$_statisticsFile);
                exit(0);
            // 重啟 workerman
            case 'restart':
            // 停止 workeran
            case 'stop':
                self::log("Workerman[$start_file] is stoping ...");
                // 想主進程發送SIGINT信號，主進程會向所有子進程發送SIGINT信號
                $master_pid && posix_kill($master_pid, SIGINT);
                // 如果 $timeout 秒後主進程沒有退出則展示失敗界面
                $timeout = 5;
                $start_time = time();
                while(1)
                {
                    // 檢查主進程是否存活
                    $master_is_alive = $master_pid && posix_kill($master_pid, 0);
                    if($master_is_alive)
                    {
                        // 檢查是否超過$timeout時間
                        if(time() - $start_time >= $timeout)
                        {
                            self::log("Workerman[$start_file] stop fail");
                            exit;
                        }
                        usleep(10000);
                        continue;
                    }
                    self::log("Workerman[$start_file] stop success");
                    // 是restart命令
                    if($command === 'stop')
                    {
                        exit(0);
                    }
                    // -d 說明是以守護進程的方式啟動
                    if($command2 === '-d')
                    {
                        Worker::$daemonize = true;
                    }
                    break;
                }
                break;
            // 平滑重啟 workerman
            case 'reload':
                posix_kill($master_pid, SIGUSR1);
                self::log("Workerman[$start_file] reload");
                exit;
            // 未知命令
            default :
                 exit("Usage: php yourfile.php {start|stop|restart|reload|status}\n");
        }
    }
    
    /**
     * 安裝信號處理函數
     * @return void
     */
    protected static function installSignal()
    {
        // stop
        pcntl_signal(SIGINT,  array('\Workerman\Worker', 'signalHandler'), false);
        // reload
        pcntl_signal(SIGUSR1, array('\Workerman\Worker', 'signalHandler'), false);
        // status
        pcntl_signal(SIGUSR2, array('\Workerman\Worker', 'signalHandler'), false);
        // ignore
        pcntl_signal(SIGPIPE, SIG_IGN, false);
    }
    
    /**
     * 為子進程重新安裝信號處理函數，使用全局事件輪詢監聽信號
     * @return void
     */
    protected static function reinstallSignal()
    {
        // uninstall stop signal handler
        pcntl_signal(SIGINT,  SIG_IGN, false);
        // uninstall reload signal handler
        pcntl_signal(SIGUSR1, SIG_IGN, false);
        // uninstall  status signal handler
        pcntl_signal(SIGUSR2, SIG_IGN, false);
        // reinstall stop signal handler
        self::$globalEvent->add(SIGINT, EventInterface::EV_SIGNAL, array('\Workerman\Worker', 'signalHandler'));
        //  uninstall  reload signal handler
        self::$globalEvent->add(SIGUSR1, EventInterface::EV_SIGNAL,array('\Workerman\Worker', 'signalHandler'));
        // uninstall  status signal handler
        self::$globalEvent->add(SIGUSR2, EventInterface::EV_SIGNAL, array('\Workerman\Worker', 'signalHandler'));
    }
    
    /**
     * 信號處理函數
     * @param int $signal
     */
    public static function signalHandler($signal)
    {
        switch($signal)
        {
            // stop
            case SIGINT:
                self::stopAll();
                break;
            // reload
            case SIGUSR1:
                self::$_pidsToRestart = self::getAllWorkerPids();;
                self::reload();
                break;
            // show status
            case SIGUSR2:
                self::writeStatisticsToStatusFile();
                break;
        }
    }

    /**
     * 嘗試以守護進程的方式運行
     * @throws Exception
     */
    protected static function daemonize()
    {
        if(!self::$daemonize)
        {
            return;
        }
        umask(0);
        $pid = pcntl_fork();
        if(-1 === $pid)
        {
            throw new Exception('fork fail');
        }
        elseif($pid > 0)
        {
            exit(0);
        }
        if(-1 === posix_setsid())
        {
            throw new Exception("setsid fail");
        }
        // fork again avoid SVR4 system regain the control of terminal
        $pid = pcntl_fork();
        if(-1 === $pid)
        {
            throw new Exception("fork fail");
        }
        elseif(0 !== $pid)
        {
            exit(0);
        }
    }

    /**
     * 重定向標準輸入輸出
     * @throws Exception
     */
    protected static function resetStd()
    {
        if(!self::$daemonize)
        {
            return;
        }
        global $STDOUT, $STDERR;
        $handle = fopen(self::$stdoutFile,"a");
        if($handle) 
        {
            unset($handle);
            @fclose(STDOUT);
            @fclose(STDERR);
            $STDOUT = fopen(self::$stdoutFile,"a");
            $STDERR = fopen(self::$stdoutFile,"a");
        }
        else
        {
            throw new Exception('can not open stdoutFile ' . self::$stdoutFile);
        }
    }
    
    /**
     * 保存pid到文件中，方便運行命令時查找主進程pid
     * @throws Exception
     */
    protected static function saveMasterPid()
    {
        self::$_masterPid = posix_getpid();
        if(false === @file_put_contents(self::$pidFile, self::$_masterPid))
        {
            throw new Exception('can not save pid to ' . self::$pidFile);
        }
    }
    
    /**
     * 獲得所有子進程的pid
     * @return array
     */
    protected static function getAllWorkerPids()
    {
        $pid_array = array(); 
        foreach(self::$_pidMap as $worker_pid_array)
        {
            foreach($worker_pid_array as $worker_pid)
            {
                $pid_array[$worker_pid] = $worker_pid;
            }
        }
        return $pid_array;
    }

    /**
     * 創建子進程
     * @return void
     */
    protected static function forkWorkers()
    {
        foreach(self::$_workers as $worker)
        {
            // 啟動過程中需要得到運行用戶名的最大長度，在status時格式化展示
            if(self::$_status === self::STATUS_STARTING)
            {
                if(empty($worker->name))
                {
                    $worker->name = $worker->getSocketName();
                }
                $worker_name_length = strlen($worker->name);
                if(self::$_maxWorkerNameLength < $worker_name_length)
                {
                    self::$_maxWorkerNameLength = $worker_name_length;
                }
            }
            
            // 創建子進程
            while(count(self::$_pidMap[$worker->workerId]) < $worker->count)
            {
                self::forkOneWorker($worker);
            }
        }
    }

    /**
     * 創建一個子進程
     * @param Worker $worker
     * @throws Exception
     */
    protected static function forkOneWorker($worker)
    {
        $pid = pcntl_fork();
        // 主進程記錄子進程pid
        if($pid > 0)
        {
            self::$_pidMap[$worker->workerId][$pid] = $pid;
        }
        // 子進程運行
        elseif(0 === $pid)
        {
            self::$_pidMap = array();
            self::$_workers = array($worker->workerId => $worker);
            Timer::delAll();
            self::setProcessTitle('WorkerMan: worker process  ' . $worker->name . ' ' . $worker->getSocketName());
            self::setProcessUser($worker->user);
            $worker->run();
            exit(250);
        }
        else
        {
            throw new Exception("forkOneWorker fail");
        }
    }
    
    /**
     * 嘗試設置運行當前進程的用戶
     * @return void
     */
    protected static function setProcessUser($user_name)
    {
        if(empty($user_name) || posix_getuid() !== 0)
        {
            return;
        }
        $user_info = posix_getpwnam($user_name);
        if($user_info['uid'] != posix_getuid() || $user_info['gid'] != posix_getgid())
        {
            if(!posix_setgid($user_info['gid']) || !posix_setuid($user_info['uid']))
            {
                self::log( 'Notice : Can not run woker as '.$user_name." , You shuld be root\n", true);
            }
        }
    }

    
    /**
     * 設置當前進程的名稱，在ps aux命令中有用
     * 注意 需要php>=5.5或者安裝了protitle擴展
     * @param string $title
     * @return void
     */
    protected static function setProcessTitle($title)
    {
        // >=php 5.5
        if (function_exists('cli_set_process_title'))
        {
            @cli_set_process_title($title);
        }
        // 需要擴展
        elseif(extension_loaded('proctitle') && function_exists('setproctitle'))
        {
            @setproctitle($title);
        }
    }
    
    /**
     * 監控所有子進程的退出事件及退出碼
     * @return void
     */
    protected static function monitorWorkers()
    {
        self::$_status = self::STATUS_RUNNING;
        while(1)
        {
            // 如果有信號到來，嘗試觸發信號處理函數
            pcntl_signal_dispatch();
            // 掛起進程，直到有子進程退出或者被信號打斷
            $status = 0;
            $pid = pcntl_wait($status, WUNTRACED);
            // 有子進程退出
            if($pid > 0)
            {
                // 查找是哪個進程組的，然後再啟動新的進程補上
                foreach(self::$_pidMap as $worker_id => $worker_pid_array)
                {
                    if(isset($worker_pid_array[$pid]))
                    {
                        $worker = self::$_workers[$worker_id];
                        // 檢查退出狀態
                        if($status !== 0)
                        {
                            self::log("worker[".$worker->name.":$pid] exit with status $status");
                        }
                       
                        // 統計，運行status命令時使用
                        if(!isset(self::$_globalStatistics['worker_exit_info'][$worker_id][$status]))
                        {
                            self::$_globalStatistics['worker_exit_info'][$worker_id][$status] = 0;
                        }
                        self::$_globalStatistics['worker_exit_info'][$worker_id][$status]++;
                        
                        // 清除子進程信息
                        unset(self::$_pidMap[$worker_id][$pid]);
                        
                        break;
                    }
                }
                // 如果不是關閉狀態，則補充新的進程
                if(self::$_status !== self::STATUS_SHUTDOWN)
                {
                    self::forkWorkers();
                    // 如果該進程是因為運行reload命令退出，則繼續執行reload流程
                    if(isset(self::$_pidsToRestart[$pid]))
                    {
                        unset(self::$_pidsToRestart[$pid]);
                        self::reload();
                    }
                }
                else
                {
                    // 如果是關閉狀態，並且所有進程退出完畢，則主進程退出
                    if(!self::getAllWorkerPids())
                    {
                        self::exitAndClearAll();
                    }
                }
            }
            else 
            {
                // 如果是關閉狀態，並且所有進程退出完畢，則主進程退出
                if(self::$_status === self::STATUS_SHUTDOWN && !self::getAllWorkerPids())
                {
                   self::exitAndClearAll();
                }
            }
        }
    }
    
    /**
     * 退出當前進程
     * @return void
     */
    protected static function exitAndClearAll()
    {
        @unlink(self::$pidFile);
        self::log("Workerman[".basename(self::$_startFile)."] has been stopped");
        exit(0);
    }
    
    /**
     * 執行平滑重啟流程
     * @return void
     */
    protected static function reload()
    {
        // 主進程部分
        if(self::$_masterPid === posix_getpid())
        {
            // 設置為平滑重啟狀態
            if(self::$_status !== self::STATUS_RELOADING && self::$_status !== self::STATUS_SHUTDOWN)
            {
                self::log("Workerman[".basename(self::$_startFile)."] reloading");
                self::$_status = self::STATUS_RELOADING;
            }
            
            // 如果有worker設置了reloadable=false，則過濾掉
            $reloadable_pid_array = array();
            foreach(self::$_pidMap as $worker_id =>$worker_pid_array)
            {
                $worker = self::$_workers[$worker_id];
                if($worker->reloadable)
                {
                    foreach($worker_pid_array as $pid)
                    {
                        $reloadable_pid_array[$pid] = $pid;
                    }
                }
            }
            
            // 得到所有可以重啟的進程
            self::$_pidsToRestart = array_intersect(self::$_pidsToRestart , $reloadable_pid_array);
            
            // 平滑重啟完畢
            if(empty(self::$_pidsToRestart))
            {
                if(self::$_status !== self::STATUS_SHUTDOWN)
                {
                    self::$_status = self::STATUS_RUNNING;
                }
                return;
            }
            // 繼續執行平滑重啟流程
            $one_worker_pid = current(self::$_pidsToRestart );
            // 給子進程發送平滑重啟信號
            posix_kill($one_worker_pid, SIGUSR1);
            // 定時器，如果子進程在KILL_WORKER_TIMER_TIME秒後沒有退出，則強行殺死
            Timer::add(self::KILL_WORKER_TIMER_TIME, 'posix_kill', array($one_worker_pid, SIGKILL), false);
        }
        // 子進程部分
        else
        {
            // 如果當前worker的reloadable屬性為真，則執行退出
            $worker = current(self::$_workers);
            if($worker->reloadable)
            {
                self::stopAll();
            }
        }
    } 
    
    /**
     * 執行關閉流程
     * @return void
     */
    public static function stopAll()
    {
        self::$_status = self::STATUS_SHUTDOWN;
        // 主進程部分
        if(self::$_masterPid === posix_getpid())
        {
            self::log("Workerman[".basename(self::$_startFile)."] Stopping ...");
            $worker_pid_array = self::getAllWorkerPids();
            // 向所有子進程發送SIGINT信號，表明關閉服務
            foreach($worker_pid_array as $worker_pid)
            {
                posix_kill($worker_pid, SIGINT);
                Timer::add(self::KILL_WORKER_TIMER_TIME, 'posix_kill', array($worker_pid, SIGKILL),false);
            }
        }
        // 子進程部分
        else
        {
            // 執行stop邏輯
            foreach(self::$_workers as $worker)
            {
                $worker->stop();
            }
            exit(0);
        }
    }
    
    /**
     * 將當前進程的統計信息寫入到統計文件
     * @return void
     */
    protected static function writeStatisticsToStatusFile()
    {
        // 主進程部分
        if(self::$_masterPid === posix_getpid())
        {
            $loadavg = sys_getloadavg();
            file_put_contents(self::$_statisticsFile, "---------------------------------------GLOBAL STATUS--------------------------------------------\n");
            file_put_contents(self::$_statisticsFile, 'Workerman version:' . Worker::VERSION . "          PHP version:".PHP_VERSION."\n", FILE_APPEND);
            file_put_contents(self::$_statisticsFile, 'start time:'. date('Y-m-d H:i:s', self::$_globalStatistics['start_timestamp']).'   run ' . floor((time()-self::$_globalStatistics['start_timestamp'])/(24*60*60)). ' days ' . floor(((time()-self::$_globalStatistics['start_timestamp'])%(24*60*60))/(60*60)) . " hours   \n", FILE_APPEND);
            file_put_contents(self::$_statisticsFile, 'load average: ' . implode(", ", $loadavg) . "\n", FILE_APPEND);
            file_put_contents(self::$_statisticsFile,  count(self::$_pidMap) . ' workers       ' . count(self::getAllWorkerPids())." processes\n", FILE_APPEND);
            file_put_contents(self::$_statisticsFile, str_pad('worker_name', self::$_maxWorkerNameLength) . " exit_status     exit_count\n", FILE_APPEND);
            foreach(self::$_pidMap as $worker_id =>$worker_pid_array)
            {
                $worker = self::$_workers[$worker_id];
                if(isset(self::$_globalStatistics['worker_exit_info'][$worker_id]))
                {
                    foreach(self::$_globalStatistics['worker_exit_info'][$worker_id] as $worker_exit_status=>$worker_exit_count)
                    {
                        file_put_contents(self::$_statisticsFile, str_pad($worker->name, self::$_maxWorkerNameLength) . " " . str_pad($worker_exit_status, 16). " $worker_exit_count\n", FILE_APPEND);
                    }
                }
                else
                {
                    file_put_contents(self::$_statisticsFile, str_pad($worker->name, self::$_maxWorkerNameLength) . " " . str_pad(0, 16). " 0\n", FILE_APPEND);
                }
            }
            file_put_contents(self::$_statisticsFile,  "---------------------------------------PROCESS STATUS-------------------------------------------\n", FILE_APPEND);
            file_put_contents(self::$_statisticsFile, "pid\tmemory  ".str_pad('listening', self::$_maxSocketNameLength)." ".str_pad('worker_name', self::$_maxWorkerNameLength)." connections ".str_pad('total_request', 13)." ".str_pad('send_fail', 9)." ".str_pad('throw_exception', 15)."\n", FILE_APPEND);
            
            chmod(self::$_statisticsFile, 0722);
            
            foreach(self::getAllWorkerPids() as $worker_pid)
            {
                posix_kill($worker_pid, SIGUSR2);
            }
            return;
        }
        
        // 子進程部分
        $worker = current(self::$_workers);
        $wrker_status_str = posix_getpid()."\t".str_pad(round(memory_get_usage(true)/(1024*1024),2)."M", 7)." " .str_pad($worker->getSocketName(), self::$_maxSocketNameLength) ." ".str_pad(($worker->name === $worker->getSocketName() ? 'none' : $worker->name), self::$_maxWorkerNameLength)." ";
        $wrker_status_str .= str_pad(ConnectionInterface::$statistics['connection_count'], 11)." ".str_pad(ConnectionInterface::$statistics['total_request'], 14)." ".str_pad(ConnectionInterface::$statistics['send_fail'],9)." ".str_pad(ConnectionInterface::$statistics['throw_exception'],15)."\n";
        file_put_contents(self::$_statisticsFile, $wrker_status_str, FILE_APPEND);
    }
    
    /**
     * 檢查錯誤
     * @return void
     */
    public static function checkErrors()
    {
        if(self::STATUS_SHUTDOWN != self::$_status)
        {
            $error_msg = "WORKER EXIT UNEXPECTED ";
            $errors = error_get_last();
            if($errors && ($errors['type'] === E_ERROR ||
                     $errors['type'] === E_PARSE ||
                     $errors['type'] === E_CORE_ERROR ||
                     $errors['type'] === E_COMPILE_ERROR || 
                     $errors['type'] === E_RECOVERABLE_ERROR ))
            {
                $error_msg .= self::getErrorType($errors['type']) . " {$errors['message']} in {$errors['file']} on line {$errors['line']}";
            }
            self::log($error_msg);
        }
    }
    
    /**
     * 獲取錯誤類型對應的意義
     * @param integer $type
     * @return string
     */
    protected static function getErrorType($type)
    {
        switch($type)
        {
            case E_ERROR: // 1 //
                return 'E_ERROR';
            case E_WARNING: // 2 //
                return 'E_WARNING';
            case E_PARSE: // 4 //
                return 'E_PARSE';
            case E_NOTICE: // 8 //
                return 'E_NOTICE';
            case E_CORE_ERROR: // 16 //
                return 'E_CORE_ERROR';
            case E_CORE_WARNING: // 32 //
                return 'E_CORE_WARNING';
            case E_COMPILE_ERROR: // 64 //
                return 'E_COMPILE_ERROR';
            case E_COMPILE_WARNING: // 128 //
                return 'E_COMPILE_WARNING';
            case E_USER_ERROR: // 256 //
                return 'E_USER_ERROR';
            case E_USER_WARNING: // 512 //
                return 'E_USER_WARNING';
            case E_USER_NOTICE: // 1024 //
                return 'E_USER_NOTICE';
            case E_STRICT: // 2048 //
                return 'E_STRICT';
            case E_RECOVERABLE_ERROR: // 4096 //
                return 'E_RECOVERABLE_ERROR';
            case E_DEPRECATED: // 8192 //
                return 'E_DEPRECATED';
            case E_USER_DEPRECATED: // 16384 //
                return 'E_USER_DEPRECATED';
        }
        return "";
    }
    
    /**
     * 記錄日誌
     * @param string $msg
     * @return void
     */
    protected static function log($msg)
    {
        $msg = $msg."\n";
        if(self::$_status === self::STATUS_STARTING || !self::$daemonize)
        {
            echo $msg;
        }
        file_put_contents(self::$logFile, date('Y-m-d H:i:s') . " " . $msg, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * worker構造函數
     * @param string $socket_name
     * @return void
     */
    public function __construct($socket_name = '', $context_option = array())
    {
        // 保存worker實例
        $this->workerId = spl_object_hash($this);
        self::$_workers[$this->workerId] = $this;
        self::$_pidMap[$this->workerId] = array();
        
        // 獲得實例化文件路徑，用於自動加載設置根目錄
        $backrace = debug_backtrace();
        $this->_appInitPath = dirname($backrace[0]['file']);
        
        // 設置socket上下文
        if($socket_name)
        {
            $this->_socketName = $socket_name;
            if(!isset($context_option['socket']['backlog']))
            {
                $context_option['socket']['backlog'] = self::DEFAUL_BACKLOG;
            }
            $this->_context = stream_context_create($context_option);
        }
    }
    
    /**
     * 監聽端口
     * @throws Exception
     */
    public function listen()
    {
        // 設置自動加載根目錄
        Autoloader::setRootPath($this->_appInitPath);
        
        if(!$this->_socketName)
        {
            return;
        }
        // 獲得應用層通訊協議以及監聽的地址
        list($scheme, $address) = explode(':', $this->_socketName, 2);
        // 如果有指定應用層協議，則檢查對應的協議類是否存在
        if($scheme != 'tcp' && $scheme != 'udp')
        {
            $scheme = ucfirst($scheme);
            $this->_protocol = '\\Protocols\\'.$scheme;
            if(!class_exists($this->_protocol))
            {
                $this->_protocol = "\\Workerman\\Protocols\\$scheme";
                if(!class_exists($this->_protocol))
                {
                    throw new Exception("class \\Protocols\\$scheme not exist");
                }
            }
        }
        elseif($scheme === 'udp')
        {
            $this->transport = 'udp';
        }
        
        // flag
        $flags =  $this->transport === 'udp' ? STREAM_SERVER_BIND : STREAM_SERVER_BIND | STREAM_SERVER_LISTEN;
        $this->_mainSocket = stream_socket_server($this->transport.":".$address, $errno, $errmsg, $flags, $this->_context);
        if(!$this->_mainSocket)
        {
            throw new Exception($errmsg);
        }
        
        // 嘗試打開tcp的keepalive，關閉TCP Nagle算法
        if(function_exists('socket_import_stream'))
        {
            $socket   = socket_import_stream($this->_mainSocket );
            @socket_set_option($socket, SOL_SOCKET, SO_KEEPALIVE, 1);
            @socket_set_option($socket, SOL_SOCKET, TCP_NODELAY, 1);
        }
        
        // 設置非阻塞
        stream_set_blocking($this->_mainSocket, 0);
        
        // 放到全局事件輪詢中監聽_mainSocket可讀事件（客戶端連接事件）
        if(self::$globalEvent)
        {
            if($this->transport !== 'udp')
            {
                self::$globalEvent->add($this->_mainSocket, EventInterface::EV_READ, array($this, 'acceptConnection'));
            }
            else
            {
                self::$globalEvent->add($this->_mainSocket,  EventInterface::EV_READ, array($this, 'acceptUdpConnection'));
            }
        }
    }
    
    /**
     * 獲得 socket name
     * @return string
     */
    public function getSocketName()
    {
        return $this->_socketName ? $this->_socketName : 'none';
    }
    
    /**
     * 運行worker實例
     */
    public function run()
    {
        // 註冊進程退出回調，用來檢查是否有錯誤
        register_shutdown_function(array("\\Workerman\\Worker", 'checkErrors'));
        
        // 設置自動加載根目錄
        Autoloader::setRootPath($this->_appInitPath);
        
        // 如果沒有全局事件輪詢，則創建一個
        if(!self::$globalEvent)
        {
            if(extension_loaded('libevent'))
            {
                self::$globalEvent = new Libevent();
            }
            else
            {
                self::$globalEvent = new Select();
            }
            // 監聽_mainSocket上的可讀事件（客戶端連接事件）
            if($this->_socketName)
            {
                if($this->transport !== 'udp')
                {
                    self::$globalEvent->add($this->_mainSocket, EventInterface::EV_READ, array($this, 'acceptConnection'));
                }
                else
                {
                    self::$globalEvent->add($this->_mainSocket,  EventInterface::EV_READ, array($this, 'acceptUdpConnection'));
                }
            }
        }
        
        // 重新安裝事件處理函數，使用全局事件輪詢監聽信號事件
        self::reinstallSignal();
        
        // 用全局事件輪詢初始化定時器
        Timer::init(self::$globalEvent);
        
        // 如果有設置進程啟動回調，則執行
        if($this->onWorkerStart)
        {
            call_user_func($this->onWorkerStart, $this);
        }
        
        // 子進程主循環
        self::$globalEvent->loop();
    }
    
    /**
     * 停止當前worker實例
     * @return void
     */
    public function stop()
    {
        // 如果有設置進程終止回調，則執行
        if($this->onWorkerStop)
        {
            call_user_func($this->onWorkerStop, $this);
        }
        // 刪除相關監聽事件，關閉_mainSocket
        self::$globalEvent->del($this->_mainSocket, EventInterface::EV_READ);
        @fclose($this->_mainSocket);
    }

    /**
     * 接收一個客戶端連接
     * @param resources $socket
     * @return void
     */
    public function acceptConnection($socket)
    {
        // 獲得客戶端連接
        $new_socket = @stream_socket_accept($socket, 0);
        // 驚群現象，忽略
        if(false === $new_socket)
        {
            return;
        }
        
        // 初始化連接對像
        $connection = new TcpConnection($new_socket);
        $connection->worker = $this;
        $connection->protocol = $this->_protocol;
        $connection->onMessage = $this->onMessage;
        $connection->onClose = $this->onClose;
        $connection->onError = $this->onError;
        $connection->onBufferDrain = $this->onBufferDrain;
        $connection->onBufferFull = $this->onBufferFull;
        $this->connections[(int)$new_socket] = $connection;
        
        // 如果有設置連接回調，則執行
        if($this->onConnect)
        {
            try
            {
                call_user_func($this->onConnect, $connection);
            }
            catch(Exception $e)
            {
                ConnectionInterface::$statistics['throw_exception']++;
                self::log($e);
            }
        }
    }
    
    /**
     * 處理udp連接（udp其實是無連接的，這裡為保證和tcp連接接口一致）
     * @param resource $socket
     */
    public function acceptUdpConnection($socket)
    {
        $recv_buffer = stream_socket_recvfrom($socket , self::MAX_UDP_PACKEG_SIZE, 0, $remote_address);
        if(false === $recv_buffer || empty($remote_address))
        {
            return false;
        }
        // 模擬一個連接對像
        $connection = new UdpConnection($socket, $remote_address);
        if($this->onMessage)
        {
            if($this->_protocol)
            {
                $parser = $this->_protocol;
                $recv_buffer = $parser::decode($recv_buffer, $connection);
            }
            ConnectionInterface::$statistics['total_request']++;
            try
            {
               call_user_func($this->onMessage, $connection, $recv_buffer);
            }
            catch(Exception $e)
            {
                ConnectionInterface::$statistics['throw_exception']++;
            }
        }
    }
}
