<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman\Protocols;

use \Workerman\Connection\ConnectionInterface;

/**
 * Protocol interface
 */
interface ProtocolInterface
{
    /**
     * 用於分包，即在接收的buffer中返回當前請求的長度（字節）
     * 如果可以在$recv_buffer中得到請求包的長度則返回長度
     * 否則返回0，表示需要更多的數據才能得到當前請求包的長度
     * 如果返回false或者負數，則代表請求不符合協議，則連接會斷開
     * @param ConnectionInterface $connection
     * @param string $recv_buffer
     * @return int|false
     */
    public static function input($recv_buffer, ConnectionInterface $connection);
    
    /**
     * 用於請求解包
     * input返回值大於0，並且WorkerMan收到了足夠的數據，則自動調用decode
     * 然後觸發onMessage回調，並將decode解碼後的數據傳遞給onMessage回調的第二個參數
     * 也就是說當收到完整的客戶端請求時，會自動調用decode解碼，無需業務代碼中手動調用
     * @param ConnectionInterface $connection
     * @param string $recv_buffer
     * @return mixed
     */
    public static function decode($recv_buffer, ConnectionInterface $connection);
    
    /**
     * 用於請求打包
     * 當需要向客戶端發送數據即調用$connection->send($data);時
     * 會自動把$data用encode打包一次，變成符合協議的數據格式，然後再發送給客戶端
     * 也就是說發送給客戶端的數據會自動encode打包，無需業務代碼中手動調用
     * @param ConnectionInterface $connection
     * @param mixed $data
     * @return string
     */
    public static function encode($data, ConnectionInterface $connection);
}
