<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman;

use \Workerman\Worker;
use \Workerman\Protocols\Http;
use \Workerman\Protocols\HttpCache;

/**
 * 
 *  基於Worker實現的一個簡單的WebServer
 *  支持靜態文件、支持文件上傳、支持POST
 *  HTTP協議
 */
class WebServer extends Worker
{
    /**
     * 默認mime類型
     * @var string
     */
    protected static $defaultMimeType = 'text/html; charset=utf-8';
    
    /**
     * 服務器名到文件路徑的轉換
     * @var array ['workerman.net'=>'/home', 'www.workerman.net'=>'home/www']
     */
    protected $serverRoot = array();
    
    /**
     * mime類型映射關係
     * @var array
     */
    protected static $mimeTypeMap = array();
    
    
    /**
     * 用來保存用戶設置的onWorkerStart回調
     * @var callback
     */
    protected $_onWorkerStart = null;
    
    /**
     * 添加站點域名與站點目錄的對應關係，類似nginx的
     * @param string $domain
     * @param string $root_path
     * @return void
     */
    public  function addRoot($domain, $root_path)
    {
        $this->serverRoot[$domain] = $root_path;
    }
    
    /**
     * 構造函數
     * @param string $socket_name
     * @param array $context_option
     */
    public function __construct($socket_name, $context_option = array())
    {
        list($scheme, $address) = explode(':', $socket_name, 2);
        parent::__construct('http:'.$address, $context_option);
        $this->name = 'WebServer';
    }
    
    /**
     * 運行
     * @see Workerman.Worker::run()
     */
    public function run()
    {
        $this->_onWorkerStart = $this->onWorkerStart;
        $this->onWorkerStart = array($this, 'onWorkerStart');
        $this->onMessage = array($this, 'onMessage');
        parent::run();
    }
    
    /**
     * 進程啟動的時候一些初始化工作
     * @throws \Exception
     */
    public function onWorkerStart()
    {
        if(empty($this->serverRoot))
        {
            throw new \Exception('server root not set, please use WebServer::addRoot($domain, $root_path) to set server root path');
        }
        // 初始化HttpCache
        HttpCache::init();
        // 初始化mimeMap
        $this->initMimeTypeMap();
        
        // 嘗試執行開發者設定的onWorkerStart回調
        if($this->_onWorkerStart)
        {
            call_user_func($this->_onWorkerStart, $this);
        }
    }
    
    /**
     * 初始化mimeType
     * @return void
     */
    public function initMimeTypeMap()
    {
        $mime_file = Http::getMimeTypesFile();
        if(!is_file($mime_file))
        {
            $this->notice("$mime_file mime.type file not fond");
            return;
        }
        $items = file($mime_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if(!is_array($items))
        {
            $this->log("get $mime_file mime.type content fail");
            return;
        }
        foreach($items as $content)
        {
            if(preg_match("/\s*(\S+)\s+(\S.+)/", $content, $match))
            {
                $mime_type = $match[1];
                $extension_var = $match[2];
                $extension_array = explode(' ', substr($extension_var, 0, -1));
                foreach($extension_array as $extension)
                {
                    self::$mimeTypeMap[$extension] = $mime_type;
                } 
            }
        }
    }
    
    /**
     * 當接收到完整的http請求後的處理邏輯
     * 1、如果請求的是以php為後綴的文件，則嘗試加載
     * 2、如果請求的url沒有後綴，則嘗試加載對應目錄的index.php
     * 3、如果請求的是非php為後綴的文件，嘗試讀取原始數據並發送
     * 4、如果請求的文件不存在，則返回404
     * @param TcpConnection $connection
     * @param mixed $data
     * @return void
     */
    public function onMessage($connection, $data)
    {
        // 請求的文件
        $url_info = parse_url($_SERVER['REQUEST_URI']);
        if(!$url_info)
        {
            Http::header('HTTP/1.1 400 Bad Request');
            return $connection->close('<h1>400 Bad Request</h1>');
        }
        
        $path = $url_info['path'];
        
        $path_info = pathinfo($path);
        $extension = isset($path_info['extension']) ? $path_info['extension'] : '' ;
        if($extension === '')
        {
            $path = ($len = strlen($path)) && $path[$len -1] === '/' ? $path.'index.php' : $path . '/index.php';
            $extension = 'php';
        }
        
        $root_dir = isset($this->serverRoot[$_SERVER['HTTP_HOST']]) ? $this->serverRoot[$_SERVER['HTTP_HOST']] : current($this->serverRoot);
        
        $file = "$root_dir/$path";
        
        // 對應的php文件不存在則直接使用根目錄的index.php
        if($extension === 'php' && !is_file($file))
        {
            $file = "$root_dir/index.php";
            if(!is_file($file))
            {
                $file = "$root_dir/index.html";
                $extension = 'html';
            }
        }
        
        // 請求的文件存在
        if(is_file($file))
        {
            // 判斷是否是站點目錄裡的文件
            if((!($request_realpath = realpath($file)) || !($root_dir_realpath = realpath($root_dir))) || 0 !== strpos($request_realpath, $root_dir_realpath))
            {
                Http::header('HTTP/1.1 400 Bad Request');
                return $connection->close('<h1>400 Bad Request</h1>');
            }
            
            $file = realpath($file);
            
            // 如果請求的是php文件
            if($extension === 'php')
            {
                $cwd = getcwd();
                chdir($root_dir);
                ini_set('display_errors', 'off');
                // 緩衝輸出
                ob_start();
                // 載入php文件
                try 
                {
                    // $_SERVER變量
                    $_SERVER['REMOTE_ADDR'] = $connection->getRemoteIp();
                    $_SERVER['REMOTE_PORT'] = $connection->getRemotePort();
                    include $file;
                }
                catch(\Exception $e) 
                {
                    // 如果不是exit
                    if($e->getMessage() != 'jump_exit')
                    {
                        echo $e;
                    }
                }
                $content = ob_get_clean();
                ini_set('display_errors', 'on');
                $connection->close($content);
                chdir($cwd);
                return ;
            }
            
            // 請求的是靜態資源文件
            if(isset(self::$mimeTypeMap[$extension]))
            {
               Http::header('Content-Type: '. self::$mimeTypeMap[$extension]);
            }
            else 
            {
                Http::header('Content-Type: '. self::$defaultMimeType);
            }
            
            // 獲取文件信息
            $info = stat($file);
            
            $modified_time = $info ? date('D, d M Y H:i:s', $info['mtime']) . ' GMT' : '';
            
            // 如果有$_SERVER['HTTP_IF_MODIFIED_SINCE']
            if(!empty($_SERVER['HTTP_IF_MODIFIED_SINCE']) && $info)
            {
                // 文件沒有更改則直接304
                if($modified_time === $_SERVER['HTTP_IF_MODIFIED_SINCE'])
                {
                    // 304
                    Http::header('HTTP/1.1 304 Not Modified');
                    // 發送給客戶端
                    return $connection->close('');
                }
            }
            
            if($modified_time)
            {
                Http::header("Last-Modified: $modified_time");
            }
            // 發送給客戶端
           return $connection->close(file_get_contents($file));
        }
        else 
        {
            // 404
            Http::header("HTTP/1.1 404 Not Found");
            return $connection->close('<html><head><title>404 頁面不存在</title></head><body><center><h3>404 Not Found</h3></center></body></html>');
        }
    }
}
