<?php
/**
 * This file is part of workerman.
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the MIT-LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @author walkor<walkor@workerman.net>
 * @copyright walkor<walkor@workerman.net>
 * @link http://www.workerman.net/
 * @license http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Workerman;
ini_set('display_errors', 'on');

use \Workerman\Events\Libevent;
use \Workerman\Events\Select;
use \Workerman\Events\EventInterface;
use \Workerman\Connection\ConnectionInterface;
use \Workerman\Connection\TcpConnection;
use \Workerman\Connection\UdpConnection;
use \Workerman\Lib\Timer;
use \Workerman\Autoloader;
use \Exception;

/**
 * 
 * @author walkor<walkor@workerman.net>
 */
class Worker
{
    /**
     * 版本號
     * @var string
     */
    const VERSION = '3.1.5';
    
    /**
     * 狀態 啟動中
     * @var int
     */
    const STATUS_STARTING = 1;
    
    /**
     * 狀態 運行中
     * @var int
     */
    const STATUS_RUNNING = 2;
    
    /**
     * 狀態 停止
     * @var int
     */
    const STATUS_SHUTDOWN = 4;
    
    /**
     * 狀態 平滑重啟中
     * @var int
     */
    const STATUS_RELOADING = 8;
    
    /**
     * 給子進程發送重啟命令 KILL_WORKER_TIMER_TIME 秒後
     * 如果對應進程仍然未重啟則強行殺死
     * @var int
     */
    const KILL_WORKER_TIMER_TIME = 1;
    
    /**
     * 默認的backlog，即內核中用於存放未被進程認領（accept）的連接隊列長度
     * @var int
     */
    const DEFAUL_BACKLOG = 1024;
    
    /**
     * udp最大包長
     * @var int
     */
    const MAX_UDP_PACKEG_SIZE = 65535;
    
    /**
     * worker的名稱，用於在運行status命令時標記進程
     * @var string
     */
    public $name = 'none';
    
    /**
     * 設置當前worker實例的進程數
     * @var int
     */
    public $count = 1;
    
    /**
     * 設置當前worker進程的運行用戶，啟動時需要root超級權限
     * @var string
     */
    public $user = '';
    
    /**
     * 當前worker進程是否可以平滑重啟 
     * @var bool
     */
    public $reloadable = true;
    
    /**
     * 當worker進程啟動時，如果設置了$onWorkerStart回調函數，則運行
     * 此鉤子函數一般用於進程啟動後初始化工作
     * @var callback
     */
    public $onWorkerStart = null;
    
    /**
     * 當有客戶端連接時，如果設置了$onConnect回調函數，則運行
     * @var callback
     */
    public $onConnect = null;
    
    /**
     * 當客戶端連接上發來數據時，如果設置了$onMessage回調，則運行
     * @var callback
     */
    public $onMessage = null;
    
    /**
     * 當客戶端的連接關閉時，如果設置了$onClose回調，則運行
     * @var callback
     */
    public $onClose = null;
    
    /**
     * 當客戶端的連接發生錯誤時，如果設置了$onError回調，則運行
     * 錯誤一般為客戶端斷開連接導致數據發送失敗、服務端的發送緩衝區滿導致發送失敗等
     * 具體錯誤碼及錯誤詳情會以參數的形式傳遞給回調，參見手冊
     * @var callback
     */
    public $onError = null;
    
    /**
     * 當連接的發送緩衝區滿時，如果設置了$onBufferFull回調，則執行
     * @var callback
     */
    public $onBufferFull = null;
    
    /**
     * 當鏈接的發送緩衝區被清空時，如果設置了$onBufferDrain回調，則執行
     * @var callback
     */
    public $onBufferDrain = null;
    
    /**
     * 當前進程退出時（由於平滑重啟或者服務停止導致），如果設置了此回調，則運行
     * @var callback
     */
    public $onWorkerStop = null;
    
    /**
     * 傳輸層協議
     * @var string
     */
    public $transport = 'tcp';
    
    /**
     * 所有的客戶端連接
     * @var array
     */
    public $connections = array();
    
    /**
     * 應用層協議，由初始化worker時指定
     * 例如 new worker('http://0.0.0.0:8080');指定使用http協議
     * @var string
     */
    protected $_protocol = '';
    
    /**
     * 當前worker實例初始化目錄位置，用於設置應用自動加載的根目錄
     * @var string
     */
    protected $_appInitPath = '';
    
    /**
     * 是否以守護進程的方式運行。運行start時加上-d參數會自動以守護進程方式運行
     * 例如 php start.php start -d
     * @var bool
     */
    public static $daemonize = false;
    
    /**
     * 重定向標準輸出，即將所有echo、var_dump等終端輸出寫到對應文件中
     * 注意 此參數只有在以守護進程方式運行時有效
     * @var string
     */
    public static $stdoutFile = '/dev/null';
    
    /**
     * pid文件的路徑及名稱
     * 例如 Worker::$pidFile = '/tmp/workerman.pid';
     * 注意 此屬性一般不必手動設置，默認會放到php臨時目錄中
     * @var string
     */
    public static $pidFile = '';
    
    /**
     * 日誌目錄，默認在workerman根目錄下，與Applications同級
     * 可以手動設置
     * 例如 Worker::$logFile = '/tmp/workerman.log';
     * @var unknown_type
     */
    public static $logFile = '';
    
    /**
     * 全局事件輪詢庫，用於監聽所有資源的可讀可寫事件
     * @var Select/Libevent
     */
    public static $globalEvent = null;
    
    /**
     * 主進程pid
     * @var int
     */
    protected static $_masterPid = 0;
    
    /**
     * 監聽的socket
     * @var stream
     */
    protected $_mainSocket = null;
    
    /**
     * socket名稱，包括應用層協議+ip+端口號，在初始化worker時設置 
     * 值類似 http://0.0.0.0:80
     * @var string
     */
    protected $_socketName = '';
    
    /**
     * socket的上下文，具體選項設置可以在初始化worker時傳遞
     * @var context
     */
    protected $_context = null;
    
    /**
     * 所有的worker實例
     * @var array
     */
    protected static $_workers = array();
    
    /**
     * 所有worker進程的pid
     * 格式為 [worker_id=>[pid=>pid, pid=>pid, ..], ..]
     * @var array
     */
    protected static $_pidMap = array();
    
    /**
     * 所有需要重啟的進程pid
     * 格式為 [pid=>pid, pid=>pid]
     * @var array
     */
    protected static $_pidsToRestart = array();
    
    /**
     * 當前worker狀態
     * @var int
     */
    protected static $_status = self::STATUS_STARTING;
    
    /**
     * 所有worke名稱(name屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxWorkerNameLength = 12;
    
    /**
     * 所有socket名稱(_socketName屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxSocketNameLength = 12;
    
    /**
     * 所有user名稱(user屬性)中的最大長度，用於在運行 status 命令時格式化輸出
     * @var int
     */
    protected static $_maxUserNameLength = 12;
    
    /**
     * 運行 status 命令時用於保存結果的文件名
     * @var string
     */
    protected static $_statisticsFile = '';
    
    /**
     * 啟動的全局入口文件
     * 例如 php start.php start ，則入口文件為start.php
     * @var string
     */
    protected static $_startFile = '';
    
    /**
     * 用來保存子進程句柄（windows）
     * @var array
     */
    protected static $_process = array();
    
    /**
     * 要執行的文件
     * @var array
     */
    protected static $_startFiles = array();
    
    /**
     * 運行所有worker實例
     * @return void
     */
    public static function runAll()
    {
        // 初始化環境變量
        self::init();
        // 解析命令
        self::parseCommand();
        // 初始化所有worker實例，主要是監聽端口
        self::initWorkers();
        // 展示啟動界面
        self::displayUI();
        // 運行所有的worker
        self::runAllWorkers();
        // 監控worker
        self::monitorWorkers();
    }
    
    /**
     * 初始化一些環境變量
     * @return void
     */
    public static function init()
    {
        $backtrace = debug_backtrace();
        self::$_startFile = $backtrace[count($backtrace)-1]['file'];
        // 沒有設置日誌文件，則生成一個默認值
        if(empty(self::$logFile))
        {
            self::$logFile = __DIR__ . '/../workerman.log';
        }
        // 標記狀態為啟動中
        self::$_status = self::STATUS_STARTING;
        // 全局事件輪詢庫
        self::$globalEvent = new Select();
        // 
        Timer::init(self::$globalEvent);
    }
    
    /**
     * 初始化所有的worker實例，主要工作為獲得格式化所需數據及監聽端口
     * @return void
     */
    protected static function initWorkers()
    {
        foreach(self::$_workers as $worker)
        {
            // 沒有設置worker名稱，則使用none代替
            if(empty($worker->name))
            {
                $worker->name = 'none';
            }
            // 獲得所有worker名稱中最大長度
            $worker_name_length = strlen($worker->name);
            if(self::$_maxWorkerNameLength < $worker_name_length)
            {
                self::$_maxWorkerNameLength = $worker_name_length;
            }
            // 獲得所有_socketName中最大長度
            $socket_name_length = strlen($worker->getSocketName());
            if(self::$_maxSocketNameLength < $socket_name_length)
            {
                self::$_maxSocketNameLength = $socket_name_length;
            }
            $user_name_length = strlen($worker->user);
            if(self::$_maxUserNameLength < $user_name_length)
            {
                self::$_maxUserNameLength = $user_name_length;
            }
        }
    }
    
    /**
     * 運行所有的worker
     */
    public static function runAllWorkers()
    {
        // 只有一個start文件時執行run
        if(count(self::$_startFiles) === 1)
        {
            // win不支持同一個頁面執初始化多個worker
            if(count(self::$_workers) > 1)
            {
                echo "@@@multi workers init in one php file are not support@@@\r\n";
            }
            elseif(count(self::$_workers) <= 0)
            {
                exit("@@@no worker inited@@@\r\n\r\n");
            }
            
            // 執行worker的run方法
            reset(self::$_workers);
            $worker = current(self::$_workers);
            $worker->listen();
            // 子進程阻塞在這裡
            $worker->run();
            exit("@@@child exit@@@\r\n");
        }
        // 多個start文件則多進程打開
        elseif(count(self::$_startFiles) > 1)
        {
            foreach(self::$_startFiles as $start_file)
            {
                self::openProcess($start_file);
            }
        }
        // 沒有start文件提示錯誤
        else
        {
            echo "@@@no worker inited@@@\r\n";
        }
    }
    
    /**
     * 打開一個子進程
     * @param string $start_file
     */
    public static function openProcess($start_file)
    {
        // 保存子進程的輸出
        $start_file = realpath($start_file);
        $std_file = sys_get_temp_dir() . '/'.str_replace(array('/', "\\", ':'), '_', $start_file).'.out.txt';
        // 將stdou stderr 重定向到文件
        $descriptorspec = array(
                0 => array('pipe', 'a'), // stdin
                1 => array('file', $std_file, 'w'), // stdout
                2 => array('file', $std_file, 'w') // stderr
        );
        
        // 保存stdin句柄，用來探測子進程是否關閉
        $pipes = array();
       
        // 打開子進程
        $process= proc_open("php $start_file -q", $descriptorspec, $pipes);
        
        // 打開stdout stderr 文件句柄
        $std_handler = fopen($std_file, 'a+');
        // 非阻塞
        stream_set_blocking($std_handler, 0);
        // 定時讀取子進程的stdout stderr
        $timer_id = Timer::add(0.1, function()use($std_handler)
        {
            echo fread($std_handler, 65535);
        });
        
        // 保存子進程句柄
        self::$_process[$start_file] = array($process, $start_file, $timer_id);
    }
    
    /**
     * 定時檢查子進程是否退出了
     */
    protected static function monitorWorkers()
    {
        // 定時檢查子進程是否退出了
        Timer::add(0.5, "\\Workerman\\Worker::checkWorkerStatus");
        
        // 主進程loop
        self::$globalEvent->loop();
    }
    
    public static function checkWorkerStatus()
    {
        foreach(self::$_process as $process_data)
        {
            $process = $process_data[0];
            $start_file = $process_data[1];
            $timer_id = $process_data[2];
            $status = proc_get_status($process);
            if(isset($status['running']))
            {
                // 子進程退出了，重啟一個子進程
                if(!$status['running'])
                {
                    echo "process $start_file terminated and try to restart\n";
                    Timer::del($timer_id);
                    @proc_close($process);
                    // 重新打開一個子進程
                    self::openProcess($start_file);
                }
            }
            else
            {
                echo "proc_get_status fail\n";
            }
        }
    }
    
    /**
     * 展示啟動界面
     * @return void
     */
    protected static function displayUI()
    {
        global $argv;
        // -q不打印
        if(in_array('-q', $argv))
        {
            return;
        }
        echo "----------------------- WORKERMAN -----------------------------\n";
        echo 'Workerman version:' . Worker::VERSION . "          PHP version:".PHP_VERSION."\n";
        echo "------------------------ WORKERS -------------------------------\n";
        echo "worker",str_pad('', self::$_maxWorkerNameLength+2-strlen('worker')), "listen",str_pad('', self::$_maxSocketNameLength+2-strlen('listen')), "processes ","status\n";
        foreach(self::$_workers as $worker)
        {
            echo str_pad($worker->name, self::$_maxWorkerNameLength+2),str_pad($worker->getSocketName(), self::$_maxSocketNameLength+2), str_pad(' '.$worker->count, 9), " [OK] \n";;
        }
        echo "----------------------------------------------------------------\n";
    }
    
    /**
     * 解析運行命令
     * php yourfile.php start | stop | restart | reload | status
     * @return void
     */
    public static function parseCommand()
    {
        global $argv;
        foreach($argv as $file)
        {
            $ext = pathinfo($file, PATHINFO_EXTENSION );
            if($ext !== 'php')
            {
                continue;
            }
            if(is_file($file))
            {
                self::$_startFiles[$file] = $file;
                include_once $file;
            }
        }
    }
    
    /**
     * 執行關閉流程
     * @return void
     */
    public static function stopAll()
    {
        self::$_status = self::STATUS_SHUTDOWN;
        exit(0);
    }
    
    /**
     * 記錄日誌
     * @param string $msg
     * @return void
     */
    protected static function log($msg)
    {
        $msg = $msg."\n";
        if(self::$_status === self::STATUS_STARTING || !self::$daemonize)
        {
            echo $msg;
        }
        file_put_contents(self::$logFile, date('Y-m-d H:i:s') . " " . $msg, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * worker構造函數
     * @param string $socket_name
     * @return void
     */
    public function __construct($socket_name = '', $context_option = array())
    {
        // 保存worker實例
        $this->workerId = spl_object_hash($this);
        self::$_workers[$this->workerId] = $this;
        self::$_pidMap[$this->workerId] = array();
        
        // 獲得實例化文件路徑，用於自動加載設置根目錄
        $backrace = debug_backtrace();
        $this->_appInitPath = dirname($backrace[0]['file']);
        
        // 設置socket上下文
        if($socket_name)
        {
            $this->_socketName = $socket_name;
            if(!isset($context_option['socket']['backlog']))
            {
                $context_option['socket']['backlog'] = self::DEFAUL_BACKLOG;
            }
            $this->_context = stream_context_create($context_option);
        }
    }
    
    /**
     * 監聽端口
     * @throws Exception
     */
    public function listen()
    {
        // 設置自動加載根目錄
        Autoloader::setRootPath($this->_appInitPath);
        
        if(!$this->_socketName)
        {
            return;
        }
        // 獲得應用層通訊協議以及監聽的地址
        list($scheme, $address) = explode(':', $this->_socketName, 2);
        // 如果有指定應用層協議，則檢查對應的協議類是否存在
        if($scheme != 'tcp' && $scheme != 'udp')
        {
            $scheme = ucfirst($scheme);
            $this->_protocol = '\\Protocols\\'.$scheme;
            if(!class_exists($this->_protocol))
            {
                $this->_protocol = "\\Workerman\\Protocols\\$scheme";
                if(!class_exists($this->_protocol))
                {
                    throw new Exception("class \\Protocols\\$scheme not exist");
                }
            }
        }
        elseif($scheme === 'udp')
        {
            $this->transport = 'udp';
        }
        
        // flag
        $flags =  $this->transport === 'udp' ? STREAM_SERVER_BIND : STREAM_SERVER_BIND | STREAM_SERVER_LISTEN;
        $this->_mainSocket = stream_socket_server($this->transport.":".$address, $errno, $errmsg, $flags, $this->_context);
        if(!$this->_mainSocket)
        {
            throw new Exception($errmsg);
        }
        
        // 嘗試打開tcp的keepalive，關閉TCP Nagle算法
        if(function_exists('socket_import_stream'))
        {
            $socket   = socket_import_stream($this->_mainSocket );
            @socket_set_option($socket, SOL_SOCKET, SO_KEEPALIVE, 1);
            @socket_set_option($socket, SOL_SOCKET, TCP_NODELAY, 1);
        }
        
        // 設置非阻塞
        stream_set_blocking($this->_mainSocket, 0);
        
        // 放到全局事件輪詢中監聽_mainSocket可讀事件（客戶端連接事件）
        if(self::$globalEvent)
        {
            if($this->transport !== 'udp')
            {
                self::$globalEvent->add($this->_mainSocket, EventInterface::EV_READ, array($this, 'acceptConnection'));
            }
            else
            {
                self::$globalEvent->add($this->_mainSocket,  EventInterface::EV_READ, array($this, 'acceptUdpConnection'));
            }
        }
    }
    
    /**
     * 獲得 socket name
     * @return string
     */
    public function getSocketName()
    {
        return $this->_socketName ? $this->_socketName : 'none';
    }
    
    /**
     * 運行worker實例
     */
    public function run()
    {
        // 設置自動加載根目錄
        Autoloader::setRootPath($this->_appInitPath);
        
        // 則創建一個全局事件輪詢
        if(extension_loaded('libevent'))
        {
            self::$globalEvent = new Libevent();
        }
        else
        {
            self::$globalEvent = new Select();
        }
        // 監聽_mainSocket上的可讀事件（客戶端連接事件）
        if($this->_socketName)
        {
            if($this->transport !== 'udp')
            {
                self::$globalEvent->add($this->_mainSocket, EventInterface::EV_READ, array($this, 'acceptConnection'));
            }
            else
            {
                self::$globalEvent->add($this->_mainSocket,  EventInterface::EV_READ, array($this, 'acceptUdpConnection'));
            }
        }
        
        // 用全局事件輪詢初始化定時器
        Timer::init(self::$globalEvent);
        
        // 如果有設置進程啟動回調，則執行
        if($this->onWorkerStart)
        {
            call_user_func($this->onWorkerStart, $this);
        }
        
        // 子進程主循環
        self::$globalEvent->loop();
    }
    
    /**
     * 停止當前worker實例
     * @return void
     */
    public function stop()
    {
        // 如果有設置進程終止回調，則執行
        if($this->onWorkerStop)
        {
            call_user_func($this->onWorkerStop, $this);
        }
        // 刪除相關監聽事件，關閉_mainSocket
        self::$globalEvent->del($this->_mainSocket, EventInterface::EV_READ);
        @fclose($this->_mainSocket);
    }

    /**
     * 接收一個客戶端連接
     * @param resources $socket
     * @return void
     */
    public function acceptConnection($socket)
    {
        // 獲得客戶端連接
        $new_socket = stream_socket_accept($socket, 0);
        
        // 驚群現象，忽略
        if(false === $new_socket)
        {
            return;
        }
        // 統計數據
        ConnectionInterface::$statistics['connection_count']++;
        // 初始化連接對像
        $connection = new TcpConnection($new_socket);
        $connection->worker = $this;
        $connection->protocol = $this->_protocol;
        $connection->onMessage = $this->onMessage;
        $connection->onClose = $this->onClose;
        $connection->onError = $this->onError;
        $connection->onBufferDrain = $this->onBufferDrain;
        $connection->onBufferFull = $this->onBufferFull;
        $this->connections[(int)$new_socket] = $connection;
        
        // 如果有設置連接回調，則執行
        if($this->onConnect)
        {
            try
            {
                call_user_func($this->onConnect, $connection);
            }
            catch(Exception $e)
            {
                ConnectionInterface::$statistics['throw_exception']++;
                self::log($e);
            }
        }
    }
    
    /**
     * 處理udp連接（udp其實是無連接的，這裡為保證和tcp連接接口一致）
     * @param resource $socket
     */
    public function acceptUdpConnection($socket)
    {
        $recv_buffer = stream_socket_recvfrom($socket , self::MAX_UDP_PACKEG_SIZE, 0, $remote_address);
        if(false === $recv_buffer || empty($remote_address))
        {
            return false;
        }
        // 模擬一個連接對像
        $connection = new UdpConnection($socket, $remote_address);
        if($this->onMessage)
        {
            if($this->_protocol)
            {
                $parser = $this->_protocol;
                $recv_buffer = $parser::decode($recv_buffer, $connection);
            }
            ConnectionInterface::$statistics['total_request']++;
            try
            {
               call_user_func($this->onMessage, $connection, $recv_buffer);
            }
            catch(Exception $e)
            {
                ConnectionInterface::$statistics['throw_exception']++;
            }
        }
    }
}
